# -*- coding: utf-8 -*-

from sandbox import sdk2
import os
import shutil
import logging
import traceback
from sandbox.projects.common.nanny import nanny
from sandbox.common.errors import TaskError
import sandbox.common.types.task as ctt


class NannyReleaseManyResources(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Скорее всего вам нужен таск NANNY_RELEASE_RESOURCE"""

    class Parameters(sdk2.Parameters):
        release_resources = sdk2.parameters.Resource(
            'Release Resources',
            description=u'Ресурсы для релиза',
            multiple=True,
            required=True,
        )

        with sdk2.parameters.Output:
            unstable = sdk2.parameters.String('unstable')
            testing = sdk2.parameters.String('testing')
            prestable = sdk2.parameters.String('prestable')
            stable = sdk2.parameters.String('stable')

        with sdk2.parameters.Group('Nanny') as nanny_block:
            release_subject = sdk2.parameters.String(
                'Nanny title',
                description=u'Название релиза'
            )
            release_comments = sdk2.parameters.String(
                'Nanny description',
                description=u'Описание релиза (обычно, changelog)',
                multiline=True,
            )
            webhook_type = nanny.WebHookTypeParameter2(
                'Webhook Type',
                description=u'Позволяет выбрать формат payload (https://nda.ya.ru/3UW7sc)',
            )
            webhook_urls = nanny.WebHookUrlParameter2(
                'Webhook URLs',
                description=u'Список URL\'ов куда Няня сделает POST-запрос после закрытия релиза',
            )
            duplicate_policy_type = nanny.DuplicatePolicyTypeParameter2(
                'Duplication Policy Type',
                description=u'Политика закрытия дубликатов в Няне (https://nda.ya.ru/3UVnKk)',
            )
            startrek_ticket_ids = nanny.StartrekTicketIdsParameter2(
                'Tracker Tickers IDs',
                description=u'Список тикетов в StartTrek в которые Няня оставит комментарии о состоянии релиза',
            )
            component = nanny.ComponentParameter2(
                'Component',
                description=u'Компонент релиза. Используется для поиска дубликатов.',
            )
            release_labels = nanny.LabelsParameter2(
                'Release Labels',
                description=u'Метки, которые будут добавлены к релизу (для фильтрации в Няне)',
            )

            with sdk2.parameters.String('Auto release') as auto_release:
                auto_release.values[""] = "-"
                for status in [ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.UNSTABLE, ctt.ReleaseStatus.PRESTABLE, ctt.ReleaseStatus.STABLE]:
                    auto_release.values[status] = status

            email_notifications_to = sdk2.parameters.List('Users to notify by email (to)', sdk2.parameters.Staff)
            email_notifications_cc = sdk2.parameters.List('Users to notify by email (cc)', sdk2.parameters.Staff)

    def get_nanny_webhook_urls(self, additional_parameters):
        return self.Parameters.webhook_urls

    def _sync_resource(self):
        """
        Publishes resource as a child of this task
        """
        for resource in self.Parameters.release_resources:
            resource_type = str(type(resource))
            attrs = dict()

            for name, value in resource.__attrs__.iteritems():
                if not (name in ['ttl', 'released']):
                    logging.info('Copy attribute {}={}'.format(name, value))
                    attrs[name] = resource.__attrs__[name]

            sdk2.Resource[resource_type](
                task=self,
                description=resource.description,
                path=self._copy_resource(resource, resource.id),
                **attrs
            )

    def _copy_resource(self, resource, resource_id):
        """
        Downloads resource to the current task folder
        """
        self.set_info('Copy resource {}'.format(resource))

        source_file = str(sdk2.ResourceData(resource).path)
        target_file = str(self.path(resource.id, os.path.basename(source_file)))

        try:
            self.set_info('Download from {} to {}'.format(source_file, target_file))
            copy_file_or_folder(source_file, target_file)
        except IOError:
            self.set_info(traceback.format_exc())
            raise TaskError('Error on copy resource {}'.format(resource))

        return target_file

    def _get_subject(self):
        if self.Parameters.release_subject:
            return self.Parameters.release_subject

        subject = []
        for resource in self.Parameters.release_resources:
            resource_type = str(type(resource))
            subject.append('{}={}'.format(resource_type, resource.id))

        return ','.join(subject)

    def on_execute(self):
        self._sync_resource()

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        if self.Parameters.auto_release:
            self.on_release(dict(
                release_status=self.Parameters.auto_release,
                releaser='',
                release_subject=self._get_subject(),
                release_comments=self.Parameters.release_comments,
                email_notifications=dict(
                    to=self.Parameters.email_notifications_to,
                    cc=self.Parameters.email_notifications_cc
                )
            ))

    def on_release(self, additional_parameters):
        """
        В additional_parameters прилетают параметры из тела запроса в ручку /release. https://nda.ya.ru/3UWCVW

        :param additional_parameters: введенные данные из формы релизов
        :type additional_parameters: dict
        """

        release_type = additional_parameters['release_status'].lower()

        tags = set(map(lambda tag: tag.lower(), self.Parameters.tags) + [release_type])

        additional_parameters['release_subject'] = self._get_subject()
        logging.info('Releasing task in {release_type} with parameters: {parameters} and tags: {tags}'.format(
            release_type=release_type,
            parameters=additional_parameters,
            tags=tags,
        ))

        self.Parameters.tags = list(tags)

        sdk2.Task.on_release(self, additional_parameters)
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)

        ids = self.Context.nanny_release_requests

        if len(ids) > 0:
            value = ids[-1]
            if release_type == 'testing':
                self.Parameters.testing = value
            elif release_type == 'unstable':
                self.Parameters.unstable = value
            elif release_type == 'prestable':
                self.Parameters.prestable = value
            elif release_type == 'stable':
                self.Parameters.stable = value
            else:
                raise ValueError('Invalid release type {}'.format(release_type))

    def get_nanny_webhook_type(self, additional_parameters):
        return 'RELEASE_ONLY'

def copy_file_or_folder(source_file, target_file):
    dirname = os.path.dirname(target_file)

    if not os.path.exists(dirname):
        os.makedirs(dirname)

    if os.path.isdir(source_file):
        shutil.copytree(source_file, target_file)
    else:
        shutil.copy(source_file, target_file)
