from sandbox import sdk2
import logging
from sandbox.projects.home.common.debuild import HomeDebuilderTask
from sandbox.projects.home.common.tools import rewriteVersions, rewriteStaticUrls
from sandbox.projects.home.resources import HomeXenialBuildagentLxc
from sandbox.sdk2.helpers import subprocess as sp
import os
from sandbox import common


class HomeStaticPackage(HomeDebuilderTask):
    class Parameters(HomeDebuilderTask.Parameters):
        container = HomeDebuilderTask.Parameters.container(resource_type=HomeXenialBuildagentLxc,
                                                           platform='linux_ubuntu_16.04_xenial')
        git_repo = HomeDebuilderTask.Parameters.git_repo(default='git@github.yandex-team.ru:morda/main.git')
        dist_repo = HomeDebuilderTask.Parameters.dist_repo(default_value='morda-xenial')

        with HomeDebuilderTask.Parameters.beta_build.value[False]:
            upload_s3 = sdk2.parameters.Bool('Upload to s3', default=True)

    def on_execute(self):
        HomeDebuilderTask.on_execute(self)

        if self.Context.cloned:
            return

        if self.Parameters.upload_s3 or self.Parameters.beta_build:
            logging.debug("Uploading to s3")
            self.set_s3_credentials()
            self.upload_to_s3()

    def set_s3_credentials(self):
        secret_key_name, secret_id_name = 'HOME_AWS_SECRET_ACCESS_KEY', "HOME_AWS_ACCESS_KEY_ID"

        try:
            access_secret_key = sdk2.Vault.data(secret_key_name)
            access_secret_id = sdk2.Vault.data(secret_id_name)
        except common.errors.VaultError as e:
            logging.exception(e)
            raise common.errors.TaskFailure("Could not retrieve awscli credentials")

        os.environ['AWS_SECRET_ACCESS_KEY'] = access_secret_key

        os.environ['AWS_ACCESS_KEY_ID'] = access_secret_id

    def get_static_package_path(self):
        version = str(self.get_deb_version())
        return self.Context.project_dir + '/debian/yandex-www-static-' + version.replace('.', '-')

    def get_static_project(self):
        return 'www'

    def get_s3_remote_root(self):
        if self.Parameters.beta_build:
            return 'home-beta'

        return 'home-static'

    def before_build(self):
        if self.Parameters.beta_build:
            version = self.get_version()
            rewriteVersions(self.Context.repo_dir, version, version)

    def get_s3_path(self):
        return ''

    def upload_to_s3(self):
        AWS_ENDPOINT = 'http://s3.mds.yandex.net'
        deb_version = str(self.get_deb_version())
        version = str(self.get_version())
        s3root = self.get_s3_remote_root()
        s3_default_root = 'home-static'
        s3_path = self.get_s3_path()
        static_project = self.get_static_project()
        prj_dir = self.get_static_package_path() + '/usr/local/www/static.yandex.net/' + static_project + '/' + deb_version

        if s3_default_root != s3root:
            rewriteStaticUrls(prj_dir,
                              '//yastatic.net/s3/' + s3_default_root + '/',
                              '//yastatic.net/s3/' + s3root + '/')

        comands = [
            'aws --endpoint-url=' + AWS_ENDPOINT + ' s3 cp --recursive ' + prj_dir + '/freeze/ s3://' + s3root + s3_path + '/_',
            'aws --endpoint-url=' + AWS_ENDPOINT + ' s3 cp --recursive ' + prj_dir + '/ s3://' + s3root + s3_path + '/' + version
        ]
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("upload_to_s3")) as pl:
            for cmd in comands:
                status = sp.Popen(cmd,
                                  shell=True,
                                  stdout=pl.stdout,
                                  stderr=sp.STDOUT,
                                  cwd=self.Context.project_dir).wait()
                if status != 0:
                    logging.error("Failed to perform '{cmd}'".format(cmd=cmd))
                    return
