from sandbox import sdk2
import itertools
import json
import logging
import sandbox.sandboxsdk.environments as environments


QUERY_TEMPLATE = '''
    $app_version = Re2::Capture(".*app_version=(?P<app_version>[0-9]+).*");
    $app_platform = Re2::Capture(".*app_platform=(?P<app_platform>(iphone|android)).*");

    select
    count(*) as hits,
    app_version
    FROM `{table}` as l
    where request like '{url_path}%' and $app_platform(request).app_platform == "{platform}"
    group by $app_version(request).app_version as app_version
    order by hits DESC
'''


QUERY_TEMPLATE_WIDGET_API = '''
    $app_version = Re2::Capture(".*app_version=(?P<app_version>[0-9]+).*");
    $from = Re2::Capture(".*from=(?P<from>[A-Za-z]+).*");

    select
    count(*) as hits,
    app_version
    FROM `{table}` as l
    where request like '{url_path}%' and $from(request).from {comparison_operation} "searchappBar"
    group by $app_version(request).app_version as app_version
    order by hits DESC
    ;
'''


MORDA_ACCESS_LOG_YT_PATH = '//home/logfeller/logs/morda-frontend-access-log/30min'
VERSIONS_CONFIG_YT_PATH = '//tmp/morda_pumpi_versions_config'

class MordaPumpiVersionsUpdater(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        yt_token = sdk2.parameters.String('Name of sdk2.Vault secret with YT token',
                                           default='robot-morda-pumpi-yt-token')
        yql_token = sdk2.parameters.String('Name of sdk2.Vault secret with YQL token')
        top_count = sdk2.parameters.Integer('Count of most popular versions that will be saved in versions config',
                                            default=5)

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
        ]

    def on_execute(self):
        from yt.wrapper import YtClient
        from yql.api.v1.client import YqlClient

        self.yt_read_logs_client = YtClient(
            proxy='hahn',
            token=sdk2.Vault.data(self.Parameters.yt_token)
        )
        self.yt_push_versions_client = YtClient(
            proxy='locke',
            token=sdk2.Vault.data(self.Parameters.yt_token)
        )
        self.yql_client = YqlClient(
            db='hahn',
            token=sdk2.Vault.data(self.Parameters.yql_token),
        )

        self._set_logs_table()

        versions_config = {
            'api_search_ios_versions': self._get_versions('/portal/api/search/2', 'iphone'),
            'api_search_android_versions': self._get_versions('/portal/api/search/2', 'android'),
            'api_yabrowser_ios_versions': self._get_versions('/portal/api/yabrowser/2', 'iphone'),
            'api_yabrowser_android_versions': self._get_versions('/portal/api/yabrowser/2', 'android'),
            'android_widget_searchappbar_versions': self._get_versions_widget_api('/android_widget_api/2', True),
            'android_widget_searchlib_versions': self._get_versions_widget_api('/android_widget_api/2', False)
        }

        logging.info('Versions config:')
        logging.info(json.dumps(versions_config, indent=4))

        self._push_versions_config(versions_config)

    def _set_logs_table(self):
        tables = self.yt_read_logs_client.list(MORDA_ACCESS_LOG_YT_PATH)
        self.logs_table = MORDA_ACCESS_LOG_YT_PATH + '/' + sorted(tables)[-1]
        logging.info('Latest logs table: ' + self.logs_table)

    def _get_versions(self, url_path, platform):
        request = self.yql_client.query(QUERY_TEMPLATE.format(table=self.logs_table, url_path=url_path, platform=platform))
        return self._run_request(request)

    def _get_versions_widget_api(self, url_path, from_searchappbar):
        request = self.yql_client.query(QUERY_TEMPLATE_WIDGET_API.format(table=self.logs_table,
                                                                         url_path=url_path,
                                                                         comparison_operation='==' if from_searchappbar else '!='))
        return self._run_request(request)

    def _run_request(self, request):
        request.run()
        for r in request.get_results():
            return list(i[1] for i in itertools.islice(r.get_iterator(), self.Parameters.top_count))

    def _push_versions_config(self, versions_config):
        self.yt_push_versions_client.set(VERSIONS_CONFIG_YT_PATH, versions_config)
