from sandbox import sdk2
import logging
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.home.common.debuild import HomeDebuilderTask, MordaTarballResource
from sandbox.projects.home.resources import PortalStocksTarball, HomeXenialBuildagentLxc
from sandbox.sdk2 import yav
import os
import sandbox.sdk2.path as spath
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.nanny import nanny
import sandbox.common.types.client as ctc


class PortalStocksResource(MordaTarballResource):

    def __init__(self, task, description=''):
        super(PortalStocksResource, self).__init__(task, PortalStocksTarball, description=description)

    def prepare_data(self):
        target_dir = 'portal-stocks'
        command = 'make tarball'
        secrets = yav.Secret("sec-01e8qh7tbnfsqn8f6a4sg13ab2", "ver-01e8qh7tbws6z7t4w8gk0x3gyx")
        tanker_token = secrets.data()["tanker_token"]
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("create_portal_stocks_data")) as pl:
            status = sp.Popen(command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT,
                cwd=self.task.Context.arcadia_dir, env=dict(os.environ, TANKER_TOKEN=tanker_token)).wait()
            if status != 0:
                raise Exception('Failed to create resource data')
            return self.task.Context.arcadia_dir + '/tarball', target_dir

    def create_resource(self):
        path, directory = self.prepare_data()
        tar_command = 'tar zcvf {} -C {} {}/'.format(self.file_name, path, directory)
        root_dir = self.task.Context.arcadia_dir + '/..'

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("create_tarball")) as pl:
            status = sp.Popen(tar_command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, cwd=root_dir).wait()
            if status != 0:
                raise Exception('Failed to create tarball')

            resource = self.resource_type(self.task, (self.description).replace('\n', '<br/>'), self.file_name, version='--')
            resource_data = sdk2.ResourceData(resource)
            resource_data.path.write_bytes(spath.Path(root_dir, self.file_name).read_bytes())
            resource_data.ready()


class BuildPortalStocks(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        dns = ctm.DnsType.DNS64
        privileged = True

    class Context(sdk2.Task.Context):
        pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Build portal-stocks'

        with sdk2.parameters.String('Auto release') as auto_release:
            for status in [ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.UNSTABLE, ctt.ReleaseStatus.PRESTABLE, ctt.ReleaseStatus.STABLE]:
                auto_release.values[status] = status
            auto_release.values[""] = "-"

        arcadia_path = sdk2.parameters.String('Arcadia path', default='arcadia:/arc/trunk/arcadia/stocks/stocks')
        arcadia_revision = sdk2.parameters.String('Arcadia revision', default=None)

    def on_execute(self):
        self.Context.arcadia_dir = Arcadia.checkout(self.Parameters.arcadia_path, 'stocks', revision=self.Parameters.arcadia_revision)
        resource = PortalStocksResource(self)
        resource.create_resource()

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        if self.Parameters.auto_release:
            nanny.ReleaseToNannyTask2.on_release(self, dict(
                releaser=self.author,
                release_status=self.Parameters.auto_release,
                release_subject="New stocks release",
                email_notifications=dict(to=[], cc=[]),
                release_comments='stocks release',
            ))

    def get_source(self):
        # changelog needs to be created from CI
        if not self.Context.source:
            self.Context.source = sp.check_output(["head -n 1 debian/changelog | awk '{print $1}'"],
                                                  shell=True,
                                                  cwd=self.Context.arcadia_dir).strip()
        return self.Context.source


class PortalStocksResourceCI(MordaTarballResource):

    def __init__(self, task, description=''):
        super(PortalStocksResourceCI, self).__init__(task, PortalStocksTarball, description=description)

    def prepare_data(self):
        target_dir = 'portal-stocks'
        command = 'make tarball'
        secrets = yav.Secret("sec-01e8qh7tbnfsqn8f6a4sg13ab2", "ver-01e8qh7tbws6z7t4w8gk0x3gyx")
        tanker_token = secrets.data()["tanker_token"]
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("create_portal_stocks_data")) as pl:
            status = sp.Popen(command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT,
                cwd=self.task.Context.arcadia_dir, env=dict(os.environ, TANKER_TOKEN=tanker_token)).wait()
            if status != 0:
                raise Exception('Failed to create resource data')
            return self.task.Context.arcadia_dir + '/tarball', target_dir

    def create_resource(self):
        path, directory = self.prepare_data()
        tar_command = 'tar zcvf {} -C {} {}/'.format(self.file_name, path, directory)
        root_dir = self.task.Context.arcadia_dir + '/..'

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("create_tarball")) as pl:
            status = sp.Popen(tar_command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, cwd=root_dir).wait()
            if status != 0:
                raise Exception('Failed to create tarball')

            resource = self.resource_type(self.task, (self.description).replace('\n', '<br/>'), self.file_name, version='--')
            resource_data = sdk2.ResourceData(resource)
            resource_data.path.write_bytes(spath.Path(root_dir, self.file_name).read_bytes())
            resource_data.ready()


class BuildPortalStocksCI(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        dns = ctm.DnsType.DNS64
        privileged = True
        cores = 4
        ram = 4096

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Context(sdk2.Task.Context):
        pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Build portal-stocks'

        with sdk2.parameters.String('Auto release') as auto_release:
            for status in [ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.UNSTABLE, ctt.ReleaseStatus.PRESTABLE, ctt.ReleaseStatus.STABLE]:
                auto_release.values[status] = status
            auto_release.values[""] = "-"

        stocks_path = sdk2.parameters.String('Stocks path', default='stocks/stocks')
        arcadia_revision = sdk2.parameters.String('Arcadia revision', default='arcadia-arc:/#trunk')

    def on_execute(self):
        assert self.Parameters.arcadia_revision
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_revision) as path_arc:
            self.Context.arcadia_dir = os.path.join(path_arc, 'stocks/stocks')
            os.chdir(self.Context.arcadia_dir)
            resource = PortalStocksResourceCI(self)
            resource.create_resource()

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        if self.Parameters.auto_release in ['stable', 'prestable', 'testing', 'unstable', 'cancelled']:
            nanny.ReleaseToNannyTask2.on_release(self, dict(
                releaser=self.author,
                release_status=self.Parameters.auto_release,
                release_subject="New stocks release",
                email_notifications=dict(to=[], cc=[]),
                release_comments='stocks release',
            ))

    def get_source(self):
        # changelog needs to be created from CI
        if not self.Context.source:
            self.Context.source = sp.check_output(["head -n 1 debian/changelog | awk '{print $1}'"],
                                                  shell=True,
                                                  cwd=self.Context.arcadia_dir).strip()
        return self.Context.source

