# coding: utf-8
from __future__ import print_function

import json
import logging
import time
from datetime import datetime

import requests

import params

SOLOMON_PARAM_PROJECT = "portalrtstat-test"
SOLOMON_PARAM_CLUSTER = "production"
SOLOMON_PARAM_SERVICE = "rtstat"
SOLOMON_URL = "https://solomon.yandex.net/api/v2/push?project={project}&cluster={cluster}&service={service}".format(
    project=SOLOMON_PARAM_PROJECT,
    cluster=SOLOMON_PARAM_CLUSTER,
    service=SOLOMON_PARAM_SERVICE,
)

# CLICKHOUSE_HOST = "morda-rt-clickhouse.n.yandex-team.ru"
CLICKHOUSE_HOST = "morda.clickhouse.error.yandex-team.ru"
CLICKHOUSE_DB_USER = "morda"

logging.basicConfig()
logger = logging.getLogger()
logger.setLevel(logging.INFO)


def make_clickhouse_request(query, clickhouse_host=CLICKHOUSE_HOST, db_user=None, db_pass=None):
    # url = 'http://{}/'.format(clickhouse_host)
    url = 'https://{}/'.format(clickhouse_host)
    if db_user and db_pass:
        url += '?user={0}&password={1}'.format(db_user, db_pass)
    query += ' FORMAT JSON'
    result = None
    try:
        # logger.info(query)
        result = requests.post(url, data=query)
        if result.text:
            return json.loads(result.text)['data']
    except Exception as msg:
        logger.error('Exception: {}'.format(str(msg)))
        logger.error('status code: {}'.format(result.status_code if result is not None else 'None'))
        logger.error('text: {}'.format(result.text if result is not None else 'None'))
        logger.error('Query: {}'.format(query))
        return None


def save_to_solomon(data, oauth):
    """
     curl "https://solomon.yandex.net/api/v2/push?project=portalrtstat-test&cluster=production&service=rtstat" \
         -X POST -H "Content-Type: application/json" -H "Authorization: OAuth {authtoken}" --data '{
         "commonLabels": {
           "host": "solomon-push"
         },
         "sensors": [
           {
             "labels": { "sensor": "param1" },
             "ts": "2019-07-16T13:14:15Z",
             "value": 456
           },
           {
             "labels": { "sensor": "param1" },
             "ts": 1563275655,
             "value": 463
           }
         ]
       }'
    """
    headers = {
        "Authorization": "OAuth {authtoken}".format(authtoken=oauth)
    }

    json_data = {
        "commonLabels": {
            "host": "solomon-push"
        },
        "sensors": data,
    }

    result = requests.post(SOLOMON_URL, json=json_data, headers=headers)
    # if result.status_code in [409]:
    #     logger.info('data is already inserted')
    if result.status_code not in [200]:
        logger.error('status code: {}'.format(result.status_code))
        logger.error('text: {}'.format(result.text))
    else:
        logger.info('data inserted')
        logger.info('text: {}'.format(result.text))


def push_param_today(sensor, by, factor, query, clickhouse_host, oauth, db_pass, verbose=False):
    nowtime = datetime.now()
    if nowtime.hour == 0 and nowtime.minute < 5:
        date = 'yesterday()'
    else:
        date = 'today()'

    query = query.format(date=date, from_time=int(time.time()) - 5 * 60, to_time=int(time.time()))
    data = make_clickhouse_request(query, clickhouse_host, CLICKHOUSE_DB_USER, db_pass)

    if data is None:
        return

    data_for_solomon = []

    if not len(data):
        logger.warning('have no data for param {}'.format('{} {}'.format(sensor, by)))
        logger.warning('query: {}'.format(query))
        return
    if 'mod' in data[0]:
        mods = set([row['mod'] for row in data])
        for mod in mods:
            row = [row for row in data if row['mod'] == mod][0]
            factor = mod or 'empty'

            logger.info('for parameter - {} {} {} time - {} value - {}'.format(
                sensor, by, row['mod'], datetime.fromtimestamp(row['unixtime']), row['value']))
            if row.get('value', False):
                labels = {
                    "sensor": sensor,
                }
                if by is not None:
                    labels[by] = factor
                else:
                    print('WARNING!')
                    print(sensor)
                    print(factor)

                data_for_solomon.append({
                    "labels": labels,
                    'ts': row['unixtime'],
                    'value': int(row['value']),
                })
    else:
        row = data[0]
        logger.info('for parameter - {} {} time - {} value - {}'.format(
            sensor, by, datetime.fromtimestamp(row['unixtime']), row['value']))
        if row.get('value', False):
            labels = {"sensor": sensor}
            if factor is not None:
                labels['factor'] = factor
            data_for_solomon.append({
                "labels": labels,
                'ts': row['unixtime'],
                'value': int(row['value']),
            })

    save_to_solomon(data_for_solomon, oauth)


def save_param_clicks(data, block_clicks):
    result = []
    timestamps = list(set([d['unixtime'] for d in data]))
    timestamps.sort(reverse=True)
    for timestamp in timestamps:
        for block in block_clicks:
            accumulator = 0
            for da in [d for d in data if d['unixtime'] == timestamp]:
                if block == 'total':
                    accumulator = accumulator + int(da['clicks'])
                elif block == '' and block == da['path']:
                    accumulator = accumulator + int(da['clicks'])
                elif block and (block == da['path'] or (block + '.') in da['path']):
                    accumulator = accumulator + int(da['clicks'])

            result.append({
                "labels": {
                    'sensor': 'clicks',
                    'block': block
                },
                'ts': timestamp,
                'value': accumulator,
            })
    return result


def push_param_clicks(block_clicks, oauth, clickhouse_host, db_pass):
    param = params.CLICKS_PARAM
    nowtime = datetime.now()
    if nowtime.hour == 0 and nowtime.minute < 5:
        date = 'yesterday()'
    else:
        date = 'today()'
    query = param['query'].format(date=date, from_time=int(time.time()) - 5 * 60, to_time=int(time.time()))

    data = make_clickhouse_request(query, clickhouse_host, CLICKHOUSE_DB_USER, db_pass)

    result = save_param_clicks(data, block_clicks)

    save_to_solomon(result, oauth)
    logger.info('done for clicks')


def make_requests(oauth, clickhouse_host=CLICKHOUSE_HOST):
    for param_name, param_data in params.PARAMS.items():
        push_param_today(param_name, param_data['query'], clickhouse_host, oauth, db_pass)


def make_requests_with_params(parameters, oauth, db_pass):
    for param_name, param_data in params.PARAMS.items():
        push_param_today(param_name, None, None, param_data['query'],
                         parameters.clickhouse_host, oauth, db_pass)

    logger.info('params.PARAMS_COMPLEX.keys(): ' + ', '.join(params.PARAMS_COMPLEX.keys()))
    query = params.PARAMS_COMPLEX['access_by_content']['query'].format(
        content=','.join(["'" + str(i) + "'" for i in parameters.content_types]))
    push_param_today('access', 'content', None, query, parameters.clickhouse_host, oauth, db_pass)

    for block in parameters.show_blocks:
        query = params.PARAMS_COMPLEX['shows']['query'].format(block_name=block)
        push_param_today('shows', None, block, query, parameters.clickhouse_host, oauth, db_pass)

    query = params.PARAMS_COMPLEX['access_by_geo']['query'].format(
        geoids=','.join([str(i) for i in parameters.geoids]))
    push_param_today('access', 'geo', None, query, parameters.clickhouse_host, oauth, db_pass)

    query = params.PARAMS_COMPLEX['access_by_browser']['query'].format(
        browser_names=','.join(["'" + str(i) + "'" for i in parameters.access_browser_names]))
    push_param_today('access', 'browser', None, query, parameters.clickhouse_host, oauth, db_pass)

    query = params.PARAMS_COMPLEX['access_by_mzone']['query'].format(
        mzones=','.join(["'" + str(i) + "'" for i in parameters.m_zones]))
    push_param_today('access', 'mzone', None, query, parameters.clickhouse_host, oauth, db_pass)

    push_param_clicks(parameters.block_clicks, oauth, parameters.clickhouse_host, db_pass)
