import json
from sys import stdout
from sandbox import sdk2
from requests import get, post
from logging import getLogger, DEBUG, INFO, StreamHandler, debug, exception


class PortalAlerts(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        debug_flag = sdk2.parameters.String('Debug', default=True)
        yasm_proxy = sdk2.parameters.String('Yasm proxy',
                                            default='http://portal-yasm.wdevx.yandex.ru:11005')
        golovan_host = sdk2.parameters.String(
            'Golovan host', default='yasm.yandex-team.ru')
        golovan_connect_timeout = sdk2.parameters.Integer(
            'Connect timeout to Golovan', default=60)

    def set_logger(self, debug_flag):
        logger = getLogger()
        if debug_flag:
            logger.setLevel(DEBUG)
        else:
            logger.setLevel(INFO)
        logger.handlers = [StreamHandler(stdout)]

    def request(self, path, data=None):
        headers = {'Content-Type': 'application/json',
                   }
        if data:
            req = post('http://{}/{}'.format(self.Parameters.golovan_host, path), headers=headers, json=data,
                       timeout=self.Parameters.golovan_connect_timeout)
        else:
            req = get('http://{}/{}'.format(self.Parameters.golovan_host, path), headers=headers,
                      timeout=self.Parameters.golovan_connect_timeout)

        try:
            response = req.json()
        except:
            response = req.text

        return response

    def get_templates(self):
        portal_alert_templates = []
        json = self.request(
            path='srvambry/tmpl/alerts/list?abc=home&fields=key')
        for template in json['response']:
            if template['key'].endswith('.template') or not template['key'].startswith('portal.'):
                continue
            portal_alert_templates.append(template['key'])

        return portal_alert_templates

    def get_alerts(self, template_name):
        json = self.request(
            path='srvambry/alerts/list?name_pattern=^{}'.format(template_name))
        alerts = []
        for alert in json['response']['result']:
            if 'origin' in alert and alert['origin'] == template_name:
                alerts.append(alert)

        return alerts

    def render_template(self, template_name):
        response = self.request(
            path='srvambry/tmpl/alerts/render_json/{}'.format(template_name))

        return response

    def replace_alerts(self, data):
        response = self.request(path='srvambry/alerts/replace', data=data)

        return response

    def create_alerts(self, alerts):
        response = {}
        for alert in alerts:
            response = self.request(path='srvambry/alerts/create', data=alert)

        return response

    def push_signal_to_yasm(self, signals):
        try:
            if len(signals) == 0:
                debug('Nothing to send to yasm')
                return
            values = [{'name': k, 'val': v} for k, v in signals.iteritems()]

            data = [{
                'ttl': 1800,
                'tags': {
                    'itype': 'portalalerts'
                },
                'values': values
            }]
            debug('Sending yasm signals:\n' + json.dumps(data, indent=4))

            response = post(
                url=self.Parameters.yasm_proxy,
                json=data,
                timeout=5
            )
            if response.status_code != 200:
                debug('Failed to send yasm metrics. Response status_code={}'.format(response.status_code))
        except Exception:
            exception('Failed to send yasm metrics')

    def on_execute(self):
        self.set_logger(debug_flag=self.Parameters.debug_flag)
        portal_alert_templates = self.get_templates()
        templates_amount = len(portal_alert_templates)
        for i, template_name in enumerate(portal_alert_templates):
            debug('\n[{}/{}] Processing template: {}'.format(i,
                  templates_amount, template_name))
            prev_alert_list = self.get_alerts(template_name)
            curr_alert_list = self.render_template(template_name)

            debug('Alerts found in the template: {}'.format(len(prev_alert_list)))
            debug('Alerts found in Golovan: {}'.format(len(curr_alert_list)))

            fail = False
            if len(curr_alert_list) < len(prev_alert_list):
                fail = True
            for prev_alert in prev_alert_list:
                for curr_alert in curr_alert_list:
                    if prev_alert['name'] == curr_alert['name']:
                        break
                else:
                    fail = True
                    debug('Missing alert: {}'.format(prev_alert['name']))

            new_alerts = []
            for curr_alert in curr_alert_list:
                for prev_alert in prev_alert_list:
                    if prev_alert['name'] == curr_alert['name']:
                        break
                else:
                    new_alerts.append(curr_alert)
                    debug('New alert: {}'.format(curr_alert['name']))

            if not fail:
                debug('OK! The template will be applied.')
                response = self.replace_alerts(
                    {'prefix': template_name, 'alerts': curr_alert_list})
                self.push_signal_to_yasm({template_name + '_fail_tttt': 0})
                debug(response)
            else:
                debug('Found missing alerts! Only new alerts will be applied!')
                self.push_signal_to_yasm({template_name + '_fail_tttt': 1})
                response = self.create_alerts(new_alerts)
                debug(response)
