# -*- coding: utf-8 -*-
import contextlib
import datetime
import logging
import os
import tarfile
import time

import sandbox.common.log as common_log
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.projects.common.vcs.aapi as aapi
import sandbox.projects.common.yappy.resources as ui_resources
import sandbox.projects.common.yappy.utils.node_manager as node_manager
import sandbox.projects.horadric.resources as horadric_resources
import sandbox.sdk2 as sdk2


class BuildHoradricUI(sdk2.Task):
    """ Build Horadric UI bundle """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30 * 60  # 30 min
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        nodejs_archive = sdk2.parameters.Resource(
            "Node JS", resource_type=ui_resources.NodeJsArchive, state=(ctr.State.READY,), required=True
        )
        npm_build_command = sdk2.parameters.String("npm build command", default="build_production")
        npm_install_env = sdk2.parameters.Dict("env variables for install command", default={})
        npm_build_env = sdk2.parameters.Dict("env variables for build command", default={})

    class Requirements(sdk2.task.Requirements):
        # NPM is not accessible in LXC containers.
        # https://st.yandex-team.ru/DEVTOOLSSUPPORT-10948#6113d072a98b6e133a839f6a
        client_tags = ctc.Tag.GENERIC  # & ~ctc.Tag.LXC & ctc.Tag.IPV4
        # NPM has troubles with IPv6.
        dns = ctm.DnsType.DNS64
        disk_space = 2 * 1024  # 2 Gb

    class Context(sdk2.Task.Context):
        sources_path = None
        dist_path = "dist"

    def on_execute(self):
        self.Context.sources_path = str(self.path("ui"))
        path, revision = aapi.ArcadiaApi.extract_path_and_revision(self.Parameters.checkout_arcadia_from_url)
        aapi.ArcadiaApi.export_svn_path(path, self.Context.sources_path, revision)
        logging.debug("Directrory %s: [%s]", self.Context.sources_path, os.listdir(self.Context.sources_path))
        self._setup_node()

        logging.debug("Start archive ui")
        with tarfile.TarFile("bundle.tar", mode="w") as tar:
            tar.add(os.path.join(self.Context.sources_path, self.Context.dist_path), self.Context.dist_path)
        sdk2.ResourceData(horadric_resources.HoradricUI(self, "Horadric UI", "bundle.tar")).ready()

    def _setup_node(self):
        logging.debug("start node setup, cwd: %s", self.Context.sources_path)
        node = node_manager.NodeManager(self, "http://npm.yandex-team.ru")
        node.setup(self.Context.sources_path, self.Parameters.nodejs_archive)

        node.install_modules(cwd=self.Context.sources_path, env=self.Parameters.npm_install_env)
        node.run_script(self.Parameters.npm_build_command, cwd=self.Context.sources_path, env=self.Parameters.npm_build_env)

    @contextlib.contextmanager
    def info_section(self, name):
        """ Copied from yappy/github """
        start = time.time()
        exception = None
        try:
            yield
        except Exception as e:
            exception = e
        finally:
            delta = datetime.timedelta(seconds=time.time() - start)
            delta -= datetime.timedelta(microseconds=delta.microseconds)

            state = 'DONE'
            message = '({}) {{}} {}'.format(delta, name)
            if exception:
                state = 'FAIL'
                message = '{}: {}: {}'.format(message, exception.__class__.__name__, exception)

            try:
                message = message.format(state)
            except Exception as e:
                message = 'info_section ({}) crashed (state={}): {}: {}'.format(name, state, e.__class__.__name__, e)

            # hide sensitive information, if any
            message = common_log.VaultFilter.filter_message_from_logger(logging.getLogger(), message)

            self.set_info(message)

            if exception:
                raise exception
