# -*- coding: utf-8 -*-
import os
import shutil
import logging
import collections
import itertools as it

from sandbox import common, sdk2, sandboxsdk


class HorizonGraph(sdk2.Resource):
    """
        Resource for AppHost graph
    """
    auto_backup = True


class HorizonSource(sdk2.Resource):
    """
        Resource for AppHost source
    """
    auto_backup = True


class CopyHorizonResources(sdk2.Task):
    """
        Copy passed via parameters graphs and sources
    """

    RESOURCE_TMP_DIR = "copied_resources"
    RESULT_RESOURCES_DIR = "result_resources"

    class Parameters(sdk2.Task.Parameters):
        graph_resources = sdk2.parameters.List("Resources to copy 'vertical;file_name;resource_rbtorrent'")
        source_resources = sdk2.parameters.List("Resources to copy 'vertical;configuration;file_name;resource_rbtorrent'")

        with sdk2.parameters.Output:
            created_resources = sdk2.parameters.Dict("Created resources: 'resource_rbtorrent':'resource_id'")

    @staticmethod
    def _get_name(file_name):
        if file_name.endswith(".json"):
            return file_name[:-5]
        return file_name

    @staticmethod
    def _iterate_over_resources(resources_list):
        for graph_file_info in resources_list:
            yield graph_file_info.split(";")

    # we should create resource before RemoteCopy start
    def on_enqueue(self):

        for vertical, file_name, rb_torrent in self._iterate_over_resources(self.Parameters.graph_resources):
            graph_name = self._get_name(file_name)

            self.Parameters.created_resources[rb_torrent] = HorizonGraph(
                self,
                "Horizon graph: {}/{}".format(vertical, graph_name),
                os.path.join(self.RESULT_RESOURCES_DIR, vertical, file_name),
                ttl="inf",
            ).id

        for vertical, configuration, file_name, rb_torrent in self._iterate_over_resources(self.Parameters.source_resources):
            source_name = self._get_name(file_name)

            self.Parameters.created_resources[rb_torrent] = HorizonSource(
                self,
                "Horizon source: {}/{}/{}".format(vertical, configuration, source_name),
                os.path.join(self.RESULT_RESOURCES_DIR, vertical, configuration, file_name),
                ttl="inf",
            ).id

    def postprocess_upload(self):
        # This method should be executed on a host, which performed an actual data upload.
        common_prefix = None
        basedir = self.ctx['deferred_upload']
        logging.info("Postprocessing HTTP upload stored at '%s'.", basedir)
        counters = collections.Counter()
        for root, dirs, files in os.walk(basedir):
            for f in files:
                f = os.path.join(root, f)
                counters["size"] += os.stat(f).st_size
            counters["files"] += len(files)
            counters["dirs"] += len(dirs)

        for fname in os.listdir(basedir):
            shutil.move(os.path.join(basedir, fname), self.abs_path(fname))
            common_prefix = (
                ''.join(c[0] for c in it.takewhile(lambda x: all(x[0] == y for y in x), it.izip(common_prefix, fname)))
                if common_prefix else
                fname
            )
        shutil.rmtree(basedir, ignore_errors=True)
        logging.info(
            "Resource's common prefixed detected as '%s', %d directories, %d files of total size %s.",
            common_prefix, counters["dirs"], counters["files"], common.utils.size2str(counters["size"])
        )
        if not common_prefix or not os.path.exists(os.path.join(self.abs_path(common_prefix))):
            raise sandboxsdk.errors.SandboxTaskFailureError("Uploaded files has no any common directory prefix.")
        self.change_resource_basename(self.ctx['result_resource_id'], common_prefix)
        if not counters["size"]:
            raise sandboxsdk.errors.SandboxTaskFailureError("Data of zero length has been uploaded.")

    def copy_resource(self, resource_name, resource_rbtorrent):
        logging.info("Copy resource: %s [%s]", resource_name, resource_rbtorrent)
        # при копировании скайнетом нужно сначала скопировать
        # во временную директорию
        # а потом уже перенести в ресурс
        copy = sandboxsdk.copy.RemoteCopySkynet(resource_rbtorrent, self.RESOURCE_TMP_DIR, log_dir=self.log_path())
        copy(fallback_to_bb=True)

    # переносим файл или директорию в ресурс
    def on_execute(self):
        import api.copier.errors

        os.makedirs(self.RESOURCE_TMP_DIR)
        os.makedirs(self.RESULT_RESOURCES_DIR)

        for vertical, file_name, rb_torrent in self._iterate_over_resources(self.Parameters.graph_resources):
            try:
                self.copy_resource(file_name, rb_torrent)
            except api.copier.errors.ResourceNotAvailable as ex:
                raise sandboxsdk.errors.SandboxTaskFailureError(ex)

            if not os.path.exists(os.path.join(self.RESOURCE_TMP_DIR, file_name)):
                self.logger.info("Error while copying from SkyNet: $s", rb_torrent)

            result_graph_dir = os.path.join(self.RESULT_RESOURCES_DIR, vertical)
            if not os.path.exists(result_graph_dir):
                os.makedirs(result_graph_dir)
            shutil.move(os.path.join(self.RESOURCE_TMP_DIR, file_name), os.path.join(result_graph_dir, file_name))

        for vertical, configuration, file_name, rb_torrent in self._iterate_over_resources(self.Parameters.source_resources):
            try:
                self.copy_resource(file_name, rb_torrent)
            except api.copier.errors.ResourceNotAvailable as ex:
                raise sandboxsdk.errors.SandboxTaskFailureError(ex)

            if not os.path.exists(os.path.join(self.RESOURCE_TMP_DIR, file_name)):
                self.logger.info("Error while copying from SkyNet: $s", rb_torrent)

            result_source_dir = os.path.join(self.RESULT_RESOURCES_DIR, vertical, configuration)
            if not os.path.exists(result_source_dir):
                os.makedirs(result_source_dir)
            shutil.move(os.path.join(self.RESOURCE_TMP_DIR, file_name), os.path.join(result_source_dir, file_name))
