# -*- coding: utf-8 -*-

import logging

import sandbox.common.types.client as ctc

from sandbox.projects.common import constants as consts
from sandbox.projects.common.build import parameters as ya_make_parameters
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.utils import svn_last_change
from sandbox import common

from sandbox.projects.hqcg import HQCG_BIN, HQCG_CONFIG, HQCG_PIPE_LOG_PARSER

logger = logging.getLogger(__name__)


class HighQualityContentGen(nanny.ReleaseToNannyTask, YaMakeTask):
    """\
        Build Hight Quality Content Generation bundle
    """

    type = 'BUILD_HQCG'

    input_parameters = [
        ya_make_parameters.ArcadiaUrl,
        ya_make_parameters.ArcadiaPatch,
        ya_make_parameters.UseArcadiaApiFuse,
        ya_make_parameters.ClearBuild,
        ya_make_parameters.StripBinaries,
        ya_make_parameters.TestParameter,
        ya_make_parameters.TestCustomParameters,
        ya_make_parameters.Sanitize,
        ya_make_parameters.DefinitionFlags,
        ya_make_parameters.CheckReturnCode,
        ya_make_parameters.KeepOn,
        ya_make_parameters.YaTimeout,
    ]

    execution_space = 90 * 1024  # 90Gb will be enough
    required_ram = 4 * 1024
    cores = 24
    client_tags = ctc.Tag.Group.LINUX

    target_path = 'quality/functionality/hqcg'
    rev_attr = 'rev'

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx['result_single_file'] = False
        self.ctx[consts.BUILD_BUNDLE_KEY] = False
        self.ctx['result_rd'] = 'HQCG Bundle'
        logger.info('context: %s', self.ctx)

    def get_build_type(self):
        return consts.RELEASE_BUILD_TYPE

    def get_resources(self):
        return {
            HQCG_BIN.name: {
                'description': HQCG_BIN.name,
                'resource_type': HQCG_BIN,
                'resource_path': 'hqcg'
            },
            HQCG_CONFIG.name: {
                'description': HQCG_CONFIG.name,
                'resource_type': HQCG_CONFIG,
                'resource_path': 'config'
            },
            HQCG_PIPE_LOG_PARSER.name: {
                'description': HQCG_PIPE_LOG_PARSER.name,
                'resource_type': HQCG_PIPE_LOG_PARSER,
                'resource_path': 'scripts'
            }
        }

    def get_targets(self):
        return [self.target_path]

    def get_arts(self):
        return [{'path': '{}/bin/hqcg'.format(self.target_path),
                 'dest': ''}]

    def get_arts_source(self):
        return [{'path': '{}/configs/production_config.json'.format(self.target_path),
                 'dest': 'config'},
                {'path': '{}/configs/prestable_config.json'.format(self.target_path),
                 'dest': 'config'},
                {'path': '{}/configs/testing_config.json'.format(self.target_path),
                 'dest': 'config'},
                {'path': '{}/scripts/push_client_log_parsing.py'.format(self.target_path),
                 'dest': 'scripts'}]

    def get_resources_attrs(self):
        if self.rev_attr in self.ctx:
            release_params = self.get_release_attributes()
            return {HQCG_BIN.name: release_params,
                    HQCG_CONFIG.name: release_params,
                    HQCG_PIPE_LOG_PARSER.name: release_params}
        return {}

    def on_execute(self):
        # use revision from URL or get last from SVN
        parts = self.ctx[consts.ARCADIA_URL_KEY].split('@')
        if len(parts) == 2:
            self.ctx[self.rev_attr] = parts[1]
        else:
            self.set_rev(self.get_last_rev())
        logger.info('on_execute(), context = {}'.format(self.ctx))
        return YaMakeTask.on_execute(self)

    def on_release(self, additional_parameters):
        is_stable_release = additional_parameters['release_status'] == 'stable'
        sanitize = self.ctx[ya_make_parameters.Sanitize.name]

        if is_stable_release and sanitize:
            message = 'Failed to release task: tried to deploy sanitizer binaries to production!'
            self.set_info(message)
            raise common.errors.ReleaseError(message)

        logger.info('on_release(), context = {}, additional_parameters = {}'.format(self.ctx, additional_parameters))
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        YaMakeTask.on_release(self, additional_parameters)

    def get_last_rev(self):
        if self.rev_attr in self.ctx:
            return self.ctx[self.rev_attr]
        url = '{}/{}'.format(self.ctx[consts.ARCADIA_URL_KEY], self.target_path)
        logger.info('get revision for {}'.format(url))
        return svn_last_change(url)

    def set_rev(self, rev):
        if self.rev_attr not in self.ctx:
            self.ctx[self.rev_attr] = rev
            self.ctx[consts.ARCADIA_URL_KEY] += '@' + rev

    def get_release_attributes(self, rev=None, rel=None, san=None):
        return {
            self.rev_attr: rev or self.ctx[self.rev_attr],
            'san': san or self.ctx[ya_make_parameters.Sanitize.name]
        }
