# -*- coding: utf-8 -*-
import os.path

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
import sandbox.projects.balancer.resources as balancer_resources
import sandbox.projects.common.build.parameters as build_parameters
import sandbox.projects.common.utils as common_utils
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.common.search.settings as media_settings
import sandbox.projects.BuildConfigGenerator as build_generator

GENERATOR_OUTPUT_SUBDIR = "generated/improxy"
IMPROXY_KEY = "improxy_resource_id"


class BalancerExecutableResourceParameter(parameters.LastReleasedResource):
    name = "balancer_executable_resource_id"
    description = "Balancer executable"
    resource_type = balancer_resources.BALANCER_EXECUTABLE


class ConfigArchiveResourceParameter(parameters.LastReleasedResource):
    name = "config_archive_resource_id"
    description = "Config archive"
    resource_type = resource_types.CONFIG_GENERATOR_GENERATED


def generate_project_ctx_key(project):
    return "generate_{}".format(project)


def generate_project_parameter(settings_cls):

    class GenerateParameter(parameters.SandboxBoolParameter):
        name = generate_project_ctx_key(settings_cls.NAME)
        description = "Generate {} configs".format(settings_cls.NAME)

    return GenerateParameter


class ImagesBuildImproxy(nanny.ReleaseToNannyTask, task.SandboxTask):
    type = "IMAGES_BUILD_IMPROXY"

    input_parameters = [BalancerExecutableResourceParameter, ConfigArchiveResourceParameter] + \
                       [generate_project_parameter(cls) for cls in media_settings.ALL_SETTINGS_CLASSES]

    cores = 1

    def __get_config_dir(self):
        return self.abs_path("configs")

    def __get_config_basename(self, name):
        return "improxy_{}.cfg".format(name)

    def __get_config_resource_path(self, name):
        return os.path.join(self.__get_config_dir(), self.__get_config_basename(name))

    def __get_balancer_resource_path(self):
        return self.abs_path("balancer")

    def __get_config_names(self):
        result = []
        for cls in media_settings.ALL_SETTINGS_CLASSES:
            if self.ctx[generate_project_ctx_key(cls.NAME)]:
                for region in cls.IMPROXY_CONFIG_LOCATIONS:
                    result.append("{}_{}".format(cls.NAME, region))
                    if hasattr(cls, 'IMPROXY_INTERNAL_CONFIG_LOCATIONS') and region in cls.IMPROXY_INTERNAL_CONFIG_LOCATIONS:
                        result.append("{}_{}_internal".format(cls.NAME, region))
        return result

    def __get_generator_info(self):
        generator_resource_id = self.ctx[ConfigArchiveResourceParameter.name]
        generator_task_id = channel.sandbox.get_resource(generator_resource_id).task_id
        generator_task = channel.sandbox.get_task(generator_task_id)
        return (generator_task.ctx[build_generator.GeneratorGitTagParameter.name], None)

    def __get_balancer_info(self):
        balancer_resource_id = self.ctx[BalancerExecutableResourceParameter.name]
        balancer_task_id = channel.sandbox.get_resource(balancer_resource_id).task_id
        balancer_task = channel.sandbox.get_task(balancer_task_id)

        balancer_svn_url = balancer_task.ctx[build_parameters.ArcadiaUrl.name]
        balancer_revision = svn.Arcadia.parse_url(balancer_svn_url).revision
        balancer_branch, balancer_tag = common_utils.branch_tag_from_svn_url(balancer_svn_url)
        return (balancer_revision, balancer_tag, balancer_branch)

    def on_enqueue(self):
        # binary
        balancer_resource_id = self.ctx[BalancerExecutableResourceParameter.name]
        balancer_resource = channel.sandbox.get_resource(balancer_resource_id)
        self.ctx[IMPROXY_KEY] = self._create_resource(self.descr, self.__get_balancer_resource_path(),
                                                      balancer_resources.BALANCER_IMPROXY_EXECUTABLE,
                                                      arch=balancer_resource.arch).id
        # configs
        config_names = self.__get_config_names()
        if not config_names:
            raise errors.SandboxTaskFailureError("You must select some of configuration options")
        for name in config_names:
            self._create_resource(self.descr + " " + name, self.__get_config_resource_path(name),
                                  media_settings.config_name2resource_type("improxy_" + name), arch="any")

    def arcadia_info(self):
        def _join(value1, value2):
            if value1 is None or value2 is None:
                return None
            return value1 + "," + value2

        return [_join(a, b) for a, b in zip(self.__get_balancer_info(), self.__get_generator_info())]

    def on_execute(self):
        # binary
        balancer_path = self.sync_resource(self.ctx[BalancerExecutableResourceParameter.name])
        paths.copy_path(balancer_path, self.__get_balancer_resource_path())

        # configs
        paths.make_folder(self.__get_config_dir(), True)
        archive_path = self.sync_resource(self.ctx[ConfigArchiveResourceParameter.name])
        work_dir = self.abs_path()
        process.run_process(["tar", "-xf", archive_path], log_prefix="tar", work_dir=work_dir)
        generated_dir = os.path.join(work_dir)
        for name in self.__get_config_names():
            paths.copy_path(os.path.join(generated_dir, GENERATOR_OUTPUT_SUBDIR, self.__get_config_basename(name)),
                            self.__get_config_resource_path(name))


__Task__ = ImagesBuildImproxy
