import os

from sandbox import sdk2
from sandbox.projects.common.dolbilka2 import DolbilkaPlanner2
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import process

import resources as heater_resources


YT_PATH_TEMPLATE = '//home/images/eventlogdata/{tier}'
YT_SCRIPT = 'yt_requests_prepare.py'


class ImagesBuildMmetaHeaterPlan(sdk2.Task):
    """
    Create resource with images metasearch dolbilka plan
    """

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup('Yt cluster') as yt_cluster:
            yt_cluster.values['arnold'] = yt_cluster.Value(value='Arnold')
            yt_cluster.values['hahn'] = yt_cluster.Value(value='Hahn', default=True)

        with sdk2.parameters.RadioGroup('DB tier') as db_tier:
            db_tier.values['ImgMmetaTier0'] = db_tier.Value(value='ImgMmetaTier0', default=True)
            db_tier.values['ImgRecommendTier0'] = db_tier.Value(value='ImgRecommendTier0')

        with sdk2.parameters.RadioGroup('Collection') as collection_type:
            collection_type.values['yandsearch'] = collection_type.Value(value='yandsearch', default=True)
            collection_type.values['imgscbir'] = collection_type.Value(value='imgscbir')

        plan_size = sdk2.parameters.Integer('Queries number', default=10000)
        min_plan_size = sdk2.parameters.Integer('Queries number threshold', default=10000)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

        cores = 1
        ram = 2 * 1024
        disk_space = 5 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        import yt.wrapper as yt

        yt.config['proxy']['url'] = self.Parameters.yt_cluster
        yt.config['token'] = sdk2.Vault.data(self.owner, 'yt_token')

        yt_path_root = YT_PATH_TEMPLATE.format(tier=self.Parameters.db_tier)

        date_table = sorted(yt.list(yt_path_root)).pop()

        self.Context.source_table = yt_path_root + '/' + date_table

        with self.memoize_stage.create_output_table(commit_on_entrance=False):
            self.Context.output_table = yt.create_temp_table(expiration_timeout=60 * 60 * 1000)

        with self.memoize_stage.fill_output_table(commit_on_entrance=False):
            process.run_process(
                cmd=[
                    '/skynet/python/bin/python',
                    os.path.join(os.path.dirname(__file__), YT_SCRIPT),
                    '--yt-proxy', self.Parameters.yt_cluster,
                    '--source-table', self.Context.source_table,
                    '--output-table', self.Context.output_table,
                    '--collection', self.Parameters.collection_type
                ],
                wait=True,
                check=True,
                environment=dict(
                    os.environ,
                    YT_TOKEN=yt.config['token']
                )
            )

        requests_filename = os.path.join(os.getcwd(), 'requests.txt')

        with open(requests_filename, 'w') as fp:
            filtered_requests = yt.read_table(
                self.Context.output_table + '[:#{}]'.format(self.Parameters.plan_size),
                format=yt.format.JsonFormat()
            )

            fp.writelines(
                row['request'] + '\n' for row in filtered_requests
            )
            assert fp.tell() > 4 * 1024 * self.Parameters.min_plan_size, 'Have got too few queries from YT'

        plan_filename = os.path.join(os.getcwd(), 'heater.mmeta.{}.{}.plan'.format(self.Parameters.db_tier, self.Parameters.collection_type))

        planner = DolbilkaPlanner2()
        planner.create_plan(
            requests_filename,
            result_path=plan_filename,
            rps=250,
            loader_type='plain',
        )

        resource = None
        if self.Parameters.collection_type == 'yandsearch' and self.Parameters.db_tier == 'ImgMmetaTier0':
            resource = sdk2.ResourceData(heater_resources.ImagesMmetaHeaterBasesearchMainPlan(
                self,
                'Images heater basesearch plan for Mmeta',
                os.path.basename(plan_filename),
                ttl=30,
                tier=self.Parameters.db_tier,
                collection=self.Parameters.collection_type
            ))
        elif self.Parameters.collection_type == 'imgscbir' and self.Parameters.db_tier == 'ImgMmetaTier0':
            resource = sdk2.ResourceData(heater_resources.ImagesMmetaHeaterBasesearchiCbirPlan(
                self,
                'Images heater basesearch plan for Mmeta',
                os.path.basename(plan_filename),
                ttl=30,
                tier=self.Parameters.db_tier,
                collection=self.Parameters.collection_type
            ))
        elif self.Parameters.collection_type == 'yandsearch' and self.Parameters.db_tier == 'ImgRecommendTier0':
            resource = sdk2.ResourceData(heater_resources.ImagesMmetaHeaterBasesearchRecommendPlan(
                self,
                'Images heater basesearch plan for Mmeta',
                os.path.basename(plan_filename),
                ttl=30,
                tier=self.Parameters.db_tier,
                collection=self.Parameters.collection_type
            ))

        if not resource:
            raise SandboxTaskFailureError('Either collection or tier is unknoswn: collection \'{}\' tier \'{}\''.format(
                    self.Parameters.collection_type,
                    self.Parameters.db_tier))

        resource.ready()
