# -*- coding: utf-8 -*-

import os
import os.path
import itertools
import time
import logging
import copy

from sandbox import sdk2

import sandbox.common.types.task as ctt

from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import task

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.decorators import retries
from sandbox.projects import resource_types
from sandbox.projects.common import constants
from sandbox.projects.common import decorators
from sandbox.projects.common import utils
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.build import ArcadiaTask as arcadia_task
from sandbox.projects.common.build import YaMake as yamake_task
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.vcs import arc
from sandbox.projects.images import resource_types as images_resource_types
from sandbox.projects.images.basesearch import resources as images_basesearch_resources
from sandbox.projects.images.daemons import resources as images_daemons_resources
from sandbox.projects.images.deployment import resources as images_deployment_resources
from sandbox.projects.images.idx_ops import resources as images_idx_ops_resources
from sandbox.projects.images.rq import resources as images_rq_resources

from sandbox.projects.images.deployment.ImagesSwitchBinary import ImagesSwitchBinary

from sandbox.projects.MediaLib.media_zk import MediaZkClient

from sandbox.projects.saas.AssembleRefreshConfigs import AssembleRefreshConfigs
from sandbox.projects.saas.common.resources import SaasRtyserverConfigsBundle

logger = logging.getLogger()

RETRIES = 10
DELAY = 30

_OPTIONAL_PREFIX = "extra_"  # Special suffix for optional components
_OUT_DIR = "out"
_GENERATE_PATH = 'generate_path'
_GENERATE_IDS = 'generate_ids'
_TARGETS = {
    # components
    'imgsearch2': (
        images_basesearch_resources.IMGSEARCH_EXECUTABLE,
        images_basesearch_resources.IMAGES_SEARCH_CONFIG,
        images_basesearch_resources.IMAGES_SEARCH_TIER1_CONFIG,
        images_basesearch_resources.IMAGES_QUICK_SEARCH_CONFIG,
        images_basesearch_resources.IMAGES_CBIR_SEARCH_CONFIG,
        images_basesearch_resources.IMAGES_CBIR_SEARCH_TIER1_CONFIG,
        images_basesearch_resources.IMAGES_CBIR_QUICK_SEARCH_CONFIG,
        images_basesearch_resources.IMAGES_ULTRA_SEARCH_CONFIG
    ),
    'cbirdaemon2': (
        images_daemons_resources.CBIR_DAEMON2_EXECUTABLE,
        images_daemons_resources.CBIR_DAEMON2_CONFIG,
        images_daemons_resources.CBIR_DAEMON2_API_CONFIG,
    ),
    _OPTIONAL_PREFIX + 'cbirdaemon2_fpga': (
        images_daemons_resources.CBIR_DAEMON2_FPGA_EXECUTABLE,
        images_daemons_resources.CBIR_DAEMON2_FPGA_CONFIG,
    ),
    _OPTIONAL_PREFIX + 'cbirdaemon2_gpu': (
        images_daemons_resources.CBIR_DAEMON2_GPU_EXECUTABLE,
        images_daemons_resources.CBIR_DAEMON2_GPU_CONFIG,
        images_daemons_resources.CBIR_DAEMON2_API_GPU_CONFIG,
    ),
    _OPTIONAL_PREFIX + 'cbirdaemon2_omp': (
        images_daemons_resources.CBIR_DAEMON2_OMP_EXECUTABLE,
        images_daemons_resources.CBIR_DAEMON2_OMP_CONFIG,
    ),
    'naildaemon': (
        images_daemons_resources.NAIL_DAEMON_EXECUTABLE,
        images_daemons_resources.NAIL_DAEMON_CONFIG,
    ),
    'rimdaemon': (
        images_daemons_resources.IMAGES_RIM_DAEMON_EXECUTABLE,
        images_daemons_resources.IMAGES_RIM_DAEMON_CONFIG,
    ),
    'rimpatchdaemon': (
        images_daemons_resources.IMAGES_RIMPATCH_DAEMON_EXECUTABLE,
        images_daemons_resources.IMAGES_RIMPATCH_DAEMON_CONFIG,
    ),
    'rqsearch': (
        images_rq_resources.IMAGES_RQ_BASESEARCH_EXECUTABLE,
        images_rq_resources.IMAGES_RQ_BASESEARCH_CONFIG,
    ),
    # tools
    'shard_checker': (
        resource_types.IMGSEARCH_SHARD_CHECKER_EXECUTABLE,
    ),
    'idx_ops_images': (
        images_idx_ops_resources.IDX_OPS_IMAGES_EXECUTABLE,
    ),
    'imagesrtyserver': (
        resource_types.IMGSEARCH_RTYSERVER_EXECUTABLE,
    ),
    'embedding_storage': (
        images_resource_types.IMAGES_EMBEDDING_STORAGE_EXECUTABLE,
    ),
    'inverted_index': (
        images_resource_types.IMAGES_INVERTED_INDEX_EXECUTABLE,
    ),
    # optional resources
    _OPTIONAL_PREFIX + "main_loop": (
        None,
        images_deployment_resources.IMAGES_MAIN_BASE_LOOP_CONFIG,
    ),
    _OPTIONAL_PREFIX + "garbage_loop": (
        None,
        images_deployment_resources.IMAGES_GARBAGE_BASE_LOOP_CONFIG,
    ),
    _OPTIONAL_PREFIX + "quick_loop": (
        None,
        images_deployment_resources.IMAGES_QUICK_BASE_LOOP_CONFIG,
    ),
    _OPTIONAL_PREFIX + "cbirdaemon_loop": (
        None,
        images_deployment_resources.IMAGES_CBIRDAEMON_LOOP_CONFIG,
    ),
    _OPTIONAL_PREFIX + "naildaemon_loop": (
        None,
        images_deployment_resources.IMAGES_NAILDAEMON_LOOP_CONFIG,
    ),
    _OPTIONAL_PREFIX + "instance_selector": (
        images_deployment_resources.IMAGES_INSTANCESEL,
    ),
}


class RegisterShardParameter(parameters.SandboxBoolParameter):
    name = 'register_shard'
    description = 'Register shard (for release builds only)'
    default_value = False


class ImagesBuildSearchBinary(nanny.ReleaseToNannyTask, yamake_task.YaMakeTask):
    type = "IMAGES_BUILD_SEARCH_BINARY"

    DEPLOY_GROUP = 'IMAGES-BASE-DEPLOY'

    DEPLOY_DASHBOARD_LINK_FMT = \
        "https://nanny.yandex-team.ru/ui/#/services/dashboards/catalog/{dashboard}/recipes/catalog/{recipe}/"

    DEPLOY_CC_USERS = [
    ]

    DEPLOY_QUEUE_ID = "IMAGES"
    DEPLOY_TASK_NAME = "IMAGES_SWITCH_BINARY"

    CIRCUIT = {}

    SAAS_CONFIGS_SVN_PATH = 'extsearch/images/saas/base/imagesrtyserver/configs/{}_configs'
    SAAS_CONFIG_TASK_ID_PARAM = '{}_saas_config_task_id'
    QUICK_SERVICE_NAME = 'quick'
    ULTRA_SERVICE_NAME = 'ultra'

    SAAS_RESOURCE_BY_SERVICE = {
        QUICK_SERVICE_NAME: images_basesearch_resources.IMGQUICK_SAAS_RTYSERVER_CONFIGS_BUNDLE,
        ULTRA_SERVICE_NAME: images_basesearch_resources.IMGULTRA_SAAS_RTYSERVER_CONFIGS_BUNDLE}

    client_tags = ctc.Tag.GENERIC & ctc.Tag.LINUX_PRECISE

    execution_space = yamake_task.YaMakeTask.execution_space + 100 * 1024

    cores = 24

    build_def_flags = ''

    input_parameters = \
        [RegisterShardParameter] + \
        arcadia_task.gen_input_params({'components': [key for key in _TARGETS] + ["saas_rtyserver_configs_bundle"]}) + \
        build_parameters.get_arcadia_params() + \
        build_parameters.get_aapi_parameters_as_default() + \
        yamake_task.ya_make_build_params() + \
        yamake_task._sandbox_task_params()

    def get_cbirdaemon_type(self):
        if self.ctx.get('build_cbirdaemon2', False):
            return "cbirdaemon2"
        if self.ctx.get('build_extra_cbirdaemon2_omp', False):
            return "cbirdaemon2_omp"
        if self.ctx.get('build_extra_cbirdaemon2_gpu', False):
            return "cbirdaemon2_gpu"
        if self.ctx.get('build_extra_cbirdaemon2_fpga', False):
            return "cbirdaemon2_fpga"
        return ""

    def is_cbirdaemon_build(self):
        return bool(self.get_cbirdaemon_type())

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _create_nanny_ticket(self, nanny_client, branch_name):
        title='Переключение на ветку {}'.format(branch_name)
        if self.is_cbirdaemon_build():
            title='Переключение сибирь-демонов на ветку {}'.format(branch_name)

        return nanny_client.create_ticket(
            queue_id='{}'.format(self.DEPLOY_QUEUE_ID),
            title=title,
            description=u'{}'.format(self.ctx.get('release_changelog', u'empty description')),
            responsible='anoh',
            urgent=True,
            copy_to=self.DEPLOY_CC_USERS
        )

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _update_nanny_ticket(self, nanny_client, text):
        nanny_client.update_ticket_status(self.ctx['nanny_ticket'], 'IN_QUEUE', text)

    def _create_deploy_ticket(self):
        branch_name = self.ctx.get('ap_arcadia_tag', 'unknown')

        # create ticket
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token')
        )
        nanny_ticket = self._create_nanny_ticket(nanny_client, branch_name)
        self.ctx['nanny_ticket'] = nanny_ticket['value']['id']
        logging.debug("Created Nanny ticket: {}".format(self.ctx['nanny_ticket']))
        self.set_info("Created Nanny ticket: <a href=https://nanny.yandex-team.ru/ui/#/t/{ticket}/>{ticket}</a>".format(
            ticket=self.ctx['nanny_ticket']),
            do_escape=False)

        zk_sync_node = ImagesSwitchBinary.ZK_SYNC_NODE
        cbirdaemon_type = self.get_cbirdaemon_type()
        if cbirdaemon_type:
            if cbirdaemon_type == "cbirdaemon2_omp":
                zk_sync_node = ImagesSwitchBinary.ZK_SYNC_NODE + '_cbirdaemon_omp'
                self.CIRCUIT = {
                    'description': u'Переключение сибирь-демонов (OMP)',
                    'dashboard': 'images-cbirdaemon',
                    'recipe': 'cbrd_deploy_binary_omp',
                    'zk_lock': '/media-services/images/flags/deployment/production_sas_imgcbrd_omp',
                    'semaphore': 'IMAGES_CBRD_DEPLOY'
                }
            elif cbirdaemon_type == "cbirdaemon2_gpu":
                zk_sync_node = ImagesSwitchBinary.ZK_SYNC_NODE + '_cbirdaemon_gpu'
                self.CIRCUIT = {
                    'description': u'Переключение сибирь-демонов (GPU)',
                    'dashboard': 'images-cbirdaemon',
                    'recipe': 'cbrd_deploy_binary_gpu',
                    'zk_lock': '/media-services/images/flags/deployment/production_sas_imgcbrd_gpu',
                    'semaphore': 'IMAGES_CBRD_DEPLOY'
                }
            else:
                zk_sync_node = ImagesSwitchBinary.ZK_SYNC_NODE + '_cbirdaemon_cpu'
                self.CIRCUIT = {
                    'description': u'Переключение сибирь-демонов',
                    'dashboard': 'images-cbirdaemon',
                    'recipe': 'cbrd_deploy_binary',
                    'zk_lock': '/media-services/images/flags/deployment/production_sas_imgcbrd_cpu',
                    'semaphore': 'IMAGES_CBRD_DEPLOY'
                }

            zk_value = str(len(self.CIRCUITS))
            with MediaZkClient() as zk:
                if zk.exists(zk_sync_node):
                    zk.set(zk_sync_node, zk_value)
                else:
                    zk.create(zk_sync_node, zk_value, makepath=True)

            # create task in draft status
            input_ctx = {
                ImagesSwitchBinary.Parameters.release_task.name: self.id,
                ImagesSwitchBinary.Parameters.nanny_dashboard_name.name: self.CIRCUIT['dashboard'],
                ImagesSwitchBinary.Parameters.nanny_dashboard_recipe.name: self.CIRCUIT['recipe'],
                ImagesSwitchBinary.Parameters.semaphore_name.name: self.CIRCUIT['semaphore'],
                ImagesSwitchBinary.Parameters.zookeeper_path.name: self.CIRCUIT['zk_lock'],
                ImagesSwitchBinary.Parameters.nanny_ticket.name: self.ctx['nanny_ticket'],
                ImagesSwitchBinary.Parameters.zk_sync_node.name: zk_sync_node
            }

            switch_task = sdk2.Task[self.DEPLOY_TASK_NAME]
            sandbox_task_id = switch_task(
                switch_task.current,
                description="Deploy branch {} to production services, dashboard {} recipe {}".format(branch_name, self.CIRCUIT['dashboard'], self.CIRCUIT['recipe']),
                owner=self.DEPLOY_GROUP,
                priority=self.priority,
                **input_ctx).id

            # set comment with link to sandbox task
            dashboard_link = self.DEPLOY_DASHBOARD_LINK_FMT.format(dashboard=self.CIRCUIT['dashboard'],
                                                                   recipe=self.CIRCUIT['recipe'])
            text = u'''{}: https://sandbox.yandex-team.ru/task/{}/
Метарецепт: {}
'''.format(self.CIRCUIT['description'], sandbox_task_id, unicode(dashboard_link, 'utf-8'))
            self._update_nanny_ticket(nanny_client, text)

    def on_release(self, additional_parameters):
        if additional_parameters['release_status'] == ctt.ReleaseStatus.STABLE:
            try:
                self._create_deploy_ticket()
            except Exception as err:
                logger.exception(err)
                self.set_info("<strong>Can't create deploy ticket.</strong>", do_escape=False)
        # Ticket integration logic
        # ========================
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        task.SandboxTask.on_release(self, additional_parameters)

    def initCtx(self):
        self.ctx['kill_timeout'] = 10 * 60 * 60

    def on_enqueue(self):
        yamake_task.YaMakeTask.on_enqueue(self)

        if self.ctx.get('components') == 'build_all':
            self.required_ram = 160 * 1024

        # YaMake task put all resources in pack dir.
        # This is incompatible with shard resources
        for art in self.get_arts_generate():
            res_id = self.create_resource(self.descr, art['path'], art['resource'], arch=sandboxapi.ARCH_ANY).id
            self.ctx.setdefault(_GENERATE_IDS, {})[art['path']] = res_id

    def _get_saas_config_tasks_id(self):
        return [self.ctx.get(self.SAAS_CONFIG_TASK_ID_PARAM.format(self.QUICK_SERVICE_NAME), None), self.ctx.get(self.SAAS_CONFIG_TASK_ID_PARAM.format(self.ULTRA_SERVICE_NAME), None)]

    def _need_to_build_saas_config(self):
        return self.ctx.get('components') == 'build_all' or self.ctx.get('build_saas_rtyserver_configs_bundle')

    def _create_saas_config_task(self, service):
        input_ctx = {
            AssembleRefreshConfigs.Parameters.create_oxygen_rt.name: False,
            AssembleRefreshConfigs.Parameters.create_oxygen_med.name: False,
            AssembleRefreshConfigs.Parameters.environments.name: ['common', 'frozen'],
            AssembleRefreshConfigs.Parameters.arcadia_path.name: svn.Arcadia.append(self.ctx[constants.ARCADIA_URL_KEY], self.SAAS_CONFIGS_SVN_PATH.format(service))
        }
        config_task = sdk2.Task[AssembleRefreshConfigs.type]
        task = config_task(
            config_task.current,
            description="Images {} saas rtyserver configs bundle".format(service),
            owner=self.DEPLOY_GROUP,
            priority=self.priority,
            **input_ctx)
        task.enqueue()
        self.ctx[self.SAAS_CONFIG_TASK_ID_PARAM.format(service)] = task.id

    def _create_saas_config_tasks(self):
        if not self._need_to_build_saas_config() or self._get_saas_config_tasks_id()[0] is not None:
            # we don't need to build config or already done it
            return
        self._create_saas_config_task(self.QUICK_SERVICE_NAME)
        self._create_saas_config_task(self.ULTRA_SERVICE_NAME)

    def _wait_saas_config(self):
        if not self._need_to_build_saas_config():
            return

        tasks_id = self._get_saas_config_tasks_id()
        if tasks_id[0] is None or tasks_id[1] is None:
            raise SandboxTaskFailureError("There is no build saas config subtasks to wait")

        return self.wait_tasks(tasks_id, tuple(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), wait_all=True)

    def _copy_saas_config(self, service, task_id):
        target = self.create_resource(
            arch='any',
            attributes={'ttl': "inf"},
            description='{} saas rtyserver configs bundle'.format(service),
            resource_path='{}_rtyserver_configs_bundle'.format(service),
            resource_type=self.SAAS_RESOURCE_BY_SERVICE[service].name
        )

        task_resources = sdk2.Resource.find(task_id=task_id).limit(1000)

        has_resource = False
        for resource in task_resources:
            res = sdk2.Resource[resource.id]
            if res.type == SaasRtyserverConfigsBundle:
                resource_data = sdk2.ResourceData(res)
                paths.copy_path(str(resource_data.path), target.path)
                has_resource = True

        if not has_resource:
            raise SandboxTaskFailureError("There is no saas config resource in subtask {}".format(str(task_id)))

        self.mark_resource_ready(target)

    def _get_saas_config_resource(self):
        if not self._need_to_build_saas_config():
            return
        tasks_id = self._get_saas_config_tasks_id()
        if tasks_id[0] is None or tasks_id[1] is None:
            raise SandboxTaskFailureError("There is no build saas config subtasks to get resource")

        self._copy_saas_config(self.QUICK_SERVICE_NAME, tasks_id[0])
        self._copy_saas_config(self.ULTRA_SERVICE_NAME, tasks_id[1])

    def do_execute(self):
        saas_config_tasks_id = self._get_saas_config_tasks_id()
        self._create_saas_config_tasks()

        if saas_config_tasks_id[0] is None and saas_config_tasks_id[1] is None:
            # check build targets
            all_build_def_flags = set()
            for target in self.__get_targets():
                for resource in target:
                    if hasattr(resource, "build_def_flags"):
                        all_build_def_flags.add(resource.build_def_flags)
                    else:
                        all_build_def_flags.add("")
            all_build_def_flags = list(all_build_def_flags)
            if len(all_build_def_flags) > 0 or not self._need_to_build_saas_config():
                if len(all_build_def_flags) != 1:
                    raise SandboxTaskFailureError("Can't build more then one target with different flags.\n{}".format(all_build_def_flags))
                self.build_def_flags = all_build_def_flags[0]
                yamake_task.YaMakeTask.do_execute(self)

            self._wait_saas_config()

        self._get_saas_config_resource()

    def get_build_def_flags(self):
        return self.build_def_flags

    def get_arts(self):
        result = []
        for resource in self.__get_bin_targets():
            result.append({
                'path': resource.arcadia_build_path,
                'dest': _OUT_DIR,
                'resource': str(resource)
            })
        return result

    def get_arts_source(self):
        result = []
        for resource in self.__get_src_targets():
            if not hasattr(resource, "generate_path"):
                result.append({
                    'path': resource.arcadia_path,
                    'dest': _OUT_DIR,
                    'resource': str(resource)
                })
        return result

    def get_arts_generate(self):
        result = []
        for resource in self.__get_src_targets():
            if hasattr(resource, "generate_path"):
                path = self.__generate_path(resource)
                result.append({
                    'path': path,
                    'dest': _OUT_DIR,
                    'resource': str(resource)
                })
        return result

    def get_targets(self):
        bin_targets = set([elt['path'] for elt in self.get_arts()])
        src_targets = set([elt['path'] for elt in self.get_arts_source()])
        return list(bin_targets.union(src_targets))

    def post_build(self, source_dir, output_dir, pack_dir):
        for resource in self.__get_src_targets():
            if hasattr(resource, "generate_path"):
                self.__generate_target(resource)

    def get_resources(self):
        result = {}
        for art in itertools.chain(self.get_arts(), self.get_arts_source()):
            base_path = os.path.basename(art['path'])
            result[base_path] = {
                "description": self.descr,
                "resource_path": os.path.join(_OUT_DIR, base_path),
                "resource_type": art['resource']
            }
        return result

    def __get_targets(self):
        for key, value in _TARGETS.iteritems():
            # cbirdaemon2 and shard_checker cannot be compiled with thinlto
            if ((not key.startswith(_OPTIONAL_PREFIX) and self.ctx.get('components') == 'build_all' and key != 'cbirdaemon2' and key != 'shard_checker') or
                    self.ctx.get('build_{}'.format(key))):
                yield value

    def __get_bin_targets(self):
        for target in self.__get_targets():
            if target[0]:
                yield target[0]

    def __get_src_targets(self):
        for target in self.__get_targets():
            for resource in target[1:]:
                yield resource

    def __generate_target(self, resource):
        with arc.Arc().mount_path(None, None, fetch_all=False) as arcadia_src_dir:
            arcadia_data_dir = self.__get_arcadia_data_dir()

            output_dir = self.__generate_path(resource)
            if not os.path.exists(output_dir):
                paths.make_folder(output_dir, delete_content=True)
                if hasattr(resource, "arcadia_data"):
                    for data_path in resource.arcadia_data:
                        paths.copy_path(os.path.join(arcadia_data_dir, data_path), output_dir)
                if hasattr(resource, "arcadia_path"):
                    for data_path in resource.arcadia_path:
                        paths.copy_path(os.path.join(arcadia_src_dir, data_path), output_dir)
                self.__shard_cmd("configure", output_dir)
                if utils.get_or_default(self.ctx, RegisterShardParameter):
                    self.__shard_cmd("register", output_dir)
                    utils.set_resource_attributes(
                        self.ctx[_GENERATE_IDS][output_dir],
                        {search_settings.SHARD_INSTANCE_ATTRIBUTE_NAME: output_dir}
                    )

    def __generate_path(self, resource):
        short_path = resource.generate_path
        if _GENERATE_PATH in self.ctx and short_path in self.ctx[_GENERATE_PATH]:
            return self.ctx[_GENERATE_PATH][short_path]

        long_path = "{}-{}".format(short_path, time.strftime("%Y%m%d-%H%M%S"))
        self.ctx.setdefault(_GENERATE_PATH, {})[short_path] = long_path
        return long_path

    @decorators.memoize
    def __get_arcadia_data_dir(self, test_data_dir="arcadia_tests_data/images"):
        url = self.ctx[constants.ARCADIA_URL_KEY]
        parsed_url = svn.Arcadia.parse_url(url)
        test_data_path = parsed_url.path.rstrip('/')
        assert test_data_path.endswith('arcadia')
        test_data_path = os.path.join(test_data_path[:-8], test_data_dir)
        test_data_url = svn.Arcadia.replace(url, path=test_data_path)
        test_data_dir_origin = svn.ArcadiaTestData.get_arcadia_test_data(self, test_data_url)
        return test_data_dir_origin

    def __shard_cmd(self, *args):
        process.run_process((self.__shard_tool(),) + args, wait=True)

    @decorators.memoize
    def __shard_tool(self):
        tool_id = utils.get_and_check_last_released_resource_id(
            resource_types.ISS_SHARDS,
            arch=sandboxapi.ARCH_ANY
        )
        return self.sync_resource(tool_id)


__Task__ = ImagesBuildSearchBinary
