# -*- encoding: utf-8 -*-

from datetime import datetime
import logging
import os
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common import utils
from tempfile import NamedTemporaryFile


class ImagesFunctionalityReportDataBinary(sdk2.Resource):
    """
    binary file, source: extsearch/images/functionality/report_data/bin
    """
    executable = True
    releasable = True
    releasers = ['anoh', 'shashkin']


class ImagesBuildSeeAlso(sdk2.Task):
    """
       Build see also block data
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

    class Parameters(sdk2.Task.Parameters):
        svn_file_path = sdk2.parameters.String(
            'Path to "see also" input file in Arcadia',
            default='arcadia:/arc/trunk/data/report/images/seealso.json'
        )
        yt_proxy = sdk2.parameters.String(
            'YT proxy',
            default='banach'
        )
        yt_home = sdk2.parameters.String(
            'YT home',
            default='//home/images-ontosearch/report_data'
        )
        yt_token_secret = sdk2.parameters.String(
            'YT token vault secret',
            default='yt_token'
        )
        trie_namespace = sdk2.parameters.String(
            'Trie namespace',
            default='images_see_also'
        )
        trie_key_semantics = sdk2.parameters.String(
            'Trie key semantics',
            default='uil,structkey'
        )
        states_to_save = sdk2.parameters.Integer(
            'Number of states to save',
            default=30
        )

    def _tool(self, resource_type):
        tool_id = utils.get_and_check_last_released_resource_id(resource_type)
        resource_path = str(sdk2.ResourceData(sdk2.Resource[tool_id]).path)
        return resource_path

    def _process(self, cmd_args, log_prefix, **kwargs):
        with sdk2.helpers.ProcessLog(self, logging.getLogger(log_prefix)) as pl:
            process = sdk2.helpers.subprocess.Popen(
                cmd_args,
                stdout=pl.stdout,
                stderr=pl.stderr,
                **kwargs
            )
            process.wait()
            if process.returncode:
                raise Exception('Process was finished unsuccessfully')

    def on_execute(self):
        input_file_path = os.path.abspath('seealso.json')
        pretrie_tsv_file_path = os.path.abspath('seealso_pretrie.tsv')

        sdk2.svn.Arcadia.export(self.Parameters.svn_file_path, input_file_path, force=True)

        report_data = self._tool('IMAGES_FUNCTIONALITY_REPORT_DATA_BINARY')
        self._process([report_data, 'images-see-also',
                       '--input-file', input_file_path,
                       '--output-file', pretrie_tsv_file_path],
                      log_prefix="see_also_parser")

        table_prefix = '{}.'.format(self.Parameters.trie_namespace)
        saas_kv_table = "{}/{}{}".format(self.Parameters.yt_home,
                                         table_prefix,
                                         datetime.now().strftime('%Y-%m-%d_%H%M'))

        token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_secret)
        with NamedTemporaryFile() as yt_token_file:
            yt_token_file.write(token)
            yt_token_file.flush()
            self._process([report_data, 'export-saas-kv',
                           '--proxy', self.Parameters.yt_proxy,
                           '--token-file', yt_token_file.name,
                           '--input-file', pretrie_tsv_file_path,
                           '--output-table', saas_kv_table,
                           '--namespace', self.Parameters.trie_namespace,
                           '--key-struct', self.Parameters.trie_key_semantics,
                           '--notify-api'],
                          log_prefix="see_also_export")

        from yt.wrapper import YtClient
        client = YtClient(proxy=self.Parameters.yt_proxy, token=token)
        states = sorted([table
                         for table in client.list(self.Parameters.yt_home, absolute=True)
                         if table.startswith(table_prefix)])
        for state in states[-self.Parameters.states_to_save:]:
            client.remove(state, recursive=True, force=True)
