# -*- coding: utf-8 -*-

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.base_search_quality import basesearch_response_parser
from sandbox.projects.common.base_search_quality.tree.node import WalkPath

import cPickle


class MiddlesearchResponses1Parameter(parameters.LastReleasedResource):
    name = 'middlesearch_responses1_resource_id'
    description = 'Middlesearch responses #1'
    resource_type = [
        resource_types.BASESEARCH_HR_RESPONSES,
        resource_types.SEARCH_HR_RESPONSES
    ]


class MiddlesearchResponses2Parameter(parameters.LastReleasedResource):
    name = 'middlesearch_responses2_resource_id'
    description = 'Middlesearch responses #2'
    resource_type = [
        resource_types.BASESEARCH_HR_RESPONSES,
        resource_types.SEARCH_HR_RESPONSES
    ]


class AllowedCommercialDataCountDiffMultiplier(parameters.SandboxFloatParameter):
    name = 'commercial_data_count_diff_multiplier'
    description = 'Allowed commercial data count difference multiplier'
    default_value = 0.01
    required = True


class ImagesCompareCommercialResponses(SandboxTask):
    """
        Compare commercial data for two middlsearch responses batches.
    """

    type = 'IMAGES_COMPARE_COMMERCIAL_RESPONSES'

    input_parameters = (
        MiddlesearchResponses1Parameter,
        MiddlesearchResponses2Parameter,
        AllowedCommercialDataCountDiffMultiplier,
    )

    def on_execute(self):
        responses1_resource_id = self.ctx[MiddlesearchResponses1Parameter.name]
        responses2_resource_id = self.ctx[MiddlesearchResponses2Parameter.name]

        responses1 = self._load_responses(responses1_resource_id)
        responses2 = self._load_responses(responses2_resource_id)

        self._compare_responses_commercial_data(responses1, responses2)

    def _load_responses(self, responses_resource_id):
        self.sync_resource(responses_resource_id)
        responses_resource = channel.sandbox.get_resource(responses_resource_id)

        return basesearch_response_parser.parse_responses(
            responses_resource.path,
            search_hr_responses=(responses_resource.type == resource_types.SEARCH_HR_RESPONSES)
        )

    def _compare_responses_commercial_data(self, responses1, responses2):
        commercial_data_count1 = self._commercial_data_count(responses1)
        commercial_data_count2 = self._commercial_data_count(responses2)
        assert commercial_data_count1 > 0
        assert commercial_data_count2 > 0

        if (abs(commercial_data_count1 - commercial_data_count2) > self.ctx[AllowedCommercialDataCountDiffMultiplier.name] * max(commercial_data_count1, commercial_data_count2)):
            eh.check_failed('Commercial data count difference exceeds permissible value!')

    def _commercial_data_count(self, responses):
        count = 0

        for response in responses:
            count += self._commercial_data_count_for_response(cPickle.loads(response))['count']

        return count

    def _commercial_data_count_for_response(self, response):
        commercial_data_info = {
            'count': 0
        }

        def ProcessGtaRelatedAttribute(attr):
            if attr.GetPropValue('key') == 'commercial_data':
                commercial_data_info['count'] += 1

        WalkPath(response, ['Grouping', 'Group', 'Document', 'ArchiveInfo', 'GtaRelatedAttribute'], ProcessGtaRelatedAttribute)

        return commercial_data_info


__Task__ = ImagesCompareCommercialResponses
