# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.projects.images import resource_types as images_resource_types
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess
import logging
import os

class ImagesCompareSearchDaemonsResponses(sdk2.Task):
    """
       Compare various search daemon responses
    """

    class Context(sdk2.Context):
        __first_run = True

    class Parameters(sdk2.Parameters):
        # TODO: use choice type
        with sdk2.parameters.String("Type") as type:
            type.values.embedding_storage = type.Value("embedding_storage")

        responses1 = sdk2.parameters.Resource('Search daemon responses #1',
                                              resource_type=images_resource_types.IMAGES_SEARCH_DAEMON_RESPONSES,
                                              required=True)
        responses2 = sdk2.parameters.Resource('Search daemon responses #2',
                                              resource_type=images_resource_types.IMAGES_SEARCH_DAEMON_RESPONSES,
                                              required=True)
        sdr_compare = sdk2.parameters.Resource('Search daemon compare executable',
                                               resource_type=images_resource_types.IMAGES_SDR_COMPARE_EXECUTABLE)

    def on_enqueue(self):
        if self.Context.__first_run is False:
            return

        if self.Parameters.sdr_compare is None:
            self.Parameters.sdr_compare = sdk2.Resource.find(type=images_resource_types.IMAGES_SDR_COMPARE_EXECUTABLE,
                                                             attrs=dict(released='stable'),
                                                             state='READY').first()

        self.Context.__first_run = False

    def init_resources(self):
        if self.Parameters.sdr_compare is None:
            raise SandboxTaskFailureError("Released stable \"Search daemon compare executable\" not found")

    def on_execute(self):
        self.init_resources()

        # Check search daemon type
        if self.Parameters.type == "embedding_storage":
            sdr_compare_argument = "EEmbeddingStorage"
        else:
            raise SandboxTaskFailureError("Unknown search daemon type: {}".format(self.Parameters.type))

        # Check responses files
        if self.Parameters.responses1.daemon_type != self.Parameters.type:
            raise SandboxTaskFailureError("Responses #1 file has type {}, required: {}".format(self.Parameters.responses1.daemon_type, self.Parameters.type))
        if self.Parameters.responses2.daemon_type != self.Parameters.type:
            raise SandboxTaskFailureError("Responses #2 file has type {}, required: {}".format(self.Parameters.responses2.daemon_type, self.Parameters.type))

        sdr_compare_executable = str(sdk2.ResourceData(self.Parameters.sdr_compare).path)
        responses1 = str(sdk2.ResourceData(self.Parameters.responses1).path)
        responses2 = str(sdk2.ResourceData(self.Parameters.responses2).path)

        diff_dir = str(self.path("diff"))
        os.mkdir(diff_dir)
        process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("sdr_compare"))
        sdr_compare_pid = subprocess.Popen([sdr_compare_executable,
                                            "-m", "EEmbeddingStorage",
                                            "-o", diff_dir,
                                            responses1,
                                            responses2], stdout=process_log.stdout, stderr=process_log.stdout)
        sdr_compare_pid.communicate()
        if sdr_compare_pid.returncode != 0:
            resource = sdk2.ResourceData(images_resource_types.IMAGES_SEARCH_DAEMON_COMPARE_RESULT(self,
                                                                                                   "Search daemon diff",
                                                                                                   diff_dir))
            resource.ready()
            raise SandboxTaskFailureError("Responses has diff")
