# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxUrlParameter

from sandbox.projects.common.search import components as search_components
from sandbox.projects.images import responses

import urlparse


MIDDLESEARCH_PARAMS = search_components.create_middlesearch_params(
    component_name="middlesearch",
    group_name="Middlesearch",
    use_int=False
)


_SAAS_CLUSTER_GROUP = "Saas base search"


class SaasBasesearchUrl(SandboxUrlParameter):
    name = 'saas_basesearch_addr'
    description = 'Url SAAS basesearch'
    required = True
    group = _SAAS_CLUSTER_GROUP


class ImagesGetSaasResponses(responses.ResponsesTask, SandboxTask):
    """
        run images meta search which looks at base saas, ask queries from meta search
    """
    type = 'IMAGES_GET_SAAS_RESPONSES'
    input_parameters = (SaasBasesearchUrl,) + MIDDLESEARCH_PARAMS.params + responses.ResponsesTask.input_parameters

    # export names in stx for easy creating subtask
    SAAS_BASE_SEARCH_URL = SaasBasesearchUrl.name

    QUERIES_SOURCE = responses.RESPONSE_SAVER_PARAMS.QueriesParameter.name
    QUERIES_LIMIT = responses.RESPONSE_SAVER_PARAMS.QueriesLimitParameter.name
    QUERIES_EMPTY_LIMIT = responses.MaxEmptyResponsesRate.name

    MIDDLESEARCH_BINARY = MIDDLESEARCH_PARAMS.Binary.name
    MIDDLESEARCH_REARANGE_DATA = MIDDLESEARCH_PARAMS.Data.name
    MIDDLESEARCH_REARANGE_INDEX = MIDDLESEARCH_PARAMS.Index.name
    MIDDLESEARCH_MODELES = MIDDLESEARCH_PARAMS.ArchiveModel.name
    MIDDLESEARCH_CONFIG = MIDDLESEARCH_PARAMS.Config.name

    RESPONSES_KEY = "resource_id_with_responses"
    RESPONSES_STATS_KEY = "stats_responses"

    @staticmethod
    def _fix_url_for_urlparse(url):
        attrs = urlparse.urlparse(url)
        if attrs.hostname is None and attrs.port is None:
            return "//" + url
        return url

    @staticmethod
    def _get_middlesearch(basesearch_addr,
                          snippetizer_addr):
        return search_components.get_middlesearch(
            params=MIDDLESEARCH_PARAMS,
            port=search_components.DEFAULT_MIDDLESEARCH_PORT,
            basesearches=[{
                'basesearch_type': 'IMAGESQUICK',
                'searchers': [(basesearch_addr.hostname, basesearch_addr.port)],
                'snippetizers': [(snippetizer_addr.hostname, snippetizer_addr.port)],
            }]
        )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        self.ctx[SaasBasesearchUrl.name] = self._fix_url_for_urlparse(self.ctx[SaasBasesearchUrl.name])
        self._create_responses_resource(resource_key=self.RESPONSES_KEY)

    def on_execute(self):
        bs_addr = urlparse.urlparse(self.ctx[SaasBasesearchUrl.name])
        with self._get_middlesearch(basesearch_addr=bs_addr, snippetizer_addr=bs_addr) as midlesearch:
            self._get_responses(target=midlesearch,
                                resource_key=self.RESPONSES_KEY,
                                stats_key=self.RESPONSES_STATS_KEY)

    @property
    def footer(self):
        bs_addr = urlparse.urlparse(self.ctx[SaasBasesearchUrl.name])
        base_search_info = "Saas BaseSearch Host={host} Port={port}".format(base_search=self.ctx[SaasBasesearchUrl.name],
                                                                            host=bs_addr.hostname,
                                                                            port=bs_addr.port)

        if self.RESPONSES_STATS_KEY in self.ctx:
            response_stats = "Respons stats {stat}".format(stat=self.ctx[self.RESPONSES_STATS_KEY])
        else:
            response_stats = "Respons stats haven't done yet"

        return "{line_first} <br> {line_second}".format(line_first=base_search_info,
                                                        line_second=response_stats)


__Task__ = ImagesGetSaasResponses
