import json

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import decorators
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common import utils
from sandbox.projects.common.differ import coloring
from sandbox.projects.common.priemka import BasePriemkaTask as priemka_task

from sandbox.projects.images import ImagesBuildSearchBinary as build_task
from sandbox.projects.images import ImagesQuickAcceptance as quick_acceptance
from sandbox.projects.images.basesearch import resources as images_basesearch_resources
from sandbox.projects.images.resources import task as resources_task

from . import compare_performance


_QUICK_ACCEPTANCE_TASK = "quick_acceprance_task"  # Context key to store quick acceptance task id
_GENERATION_TASKS = "generation_tasks"  # Context key to store plan generation task ids
_PERFORMANCE_TASKS = 'analyze_performance_tests'  # Context key to store performance task ids

_BUILD_TASK_TYPE = build_task.ImagesBuildSearchBinary.type


class BuildTaskId(parameters.TaskSelector):
    name = priemka_task.BasePriemkaTask.BUILD_TASK_KEY
    description = 'Build task id'
    task_type = 'IMAGES_BUILD_SEARCH_BINARY'
    required = False


class BuildImgsearchParameter(parameters.SandboxBoolParameter):
    name = 'build_imgsearch'
    description = 'Build imgsearch'
    default_value = True


class BuildRimdaemonParameter(parameters.SandboxBoolParameter):
    name = 'build_rimdaemon'
    description = 'Build rimdaemon'
    default_value = True


class BuildRqsearchParameter(parameters.SandboxBoolParameter):
    name = 'build_rqsearch'
    description = 'Build rqsearch'
    default_value = True


class BuildQuickParameter(parameters.SandboxBoolParameter):
    name = 'build_imagesrtyserver'
    description = 'Build imagesrtyserver'
    default_value = True


class ImagesPriemkaBasesearchBinary(priemka_task.BasePriemkaTask,
                                    resources_task.ImagesProductionResourcesTask,
                                    resources_task.GenerateResourcesTask,
                                    compare_performance.ComparePerformanceTask):
    """
        Creates release tag and built basesearch binary from this tag.
        Launch tests to compare performance with current production binary.
    """

    type = 'IMAGES_PRIEMKA_BASESEARCH_BINARY'

    ARCADIA_PROJECT = priemka_task.ArcadiaProject('images/base')

    input_parameters = (
        BuildTaskId,
        BuildImgsearchParameter,
        BuildRimdaemonParameter,
        BuildRqsearchParameter,
        BuildQuickParameter,
    ) + priemka_task.generate_priemka_parameters(_BUILD_TASK_TYPE)

    __HEADER_TEMPLATE = [
        "Test",
        "Task id",
        "Task status",
    ]

    def get_project(self):
        return self.ARCADIA_PROJECT

    @classmethod
    def get_build_task_type(cls):
        return _BUILD_TASK_TYPE

    # TODO: Use "build_all"
    def get_build_task_ctx(self):
        ctx = {}
        if utils.get_or_default(self.ctx, BuildImgsearchParameter):
            ctx.update({
                "build_imgsearch2": True,
                "build_shard_checker": True,
                "build_idx_ops_images": True,
            })
        if utils.get_or_default(self.ctx, BuildRimdaemonParameter):
            ctx["build_rimdaemon"] = True
        if utils.get_or_default(self.ctx, BuildRqsearchParameter):
            ctx["build_rqsearch"] = True
        if utils.get_or_default(self.ctx, BuildQuickParameter):
            ctx["build_imagesrtyserver"] = True
            ctx["build_saas_rtyserver_configs_bundle"] = True
        return ctx

    def get_main_resource_type(self):
        return images_basesearch_resources.IMGSEARCH_EXECUTABLE.name

    @property
    def footer(self):
        items = [{
            "<h4>Performance tests<h4>": self._performance_footer(_PERFORMANCE_TASKS),
        }]

        changelog_resource = self.get_plain_text_changelog_resource()
        if changelog_resource:
            changelog_template = "<a href='//proxy.sandbox.yandex-team.ru/{0}'>{0}</a>".format(changelog_resource)
        else:
            changelog_template = "<span style='color:red'>No changelog found, see logs for details</span>"

        items.append({
            "<h4>Other</h4>": {
                "header": [
                    {"key": "k1", "title": "&nbsp;"},
                    {"key": "k2", "title": "&nbsp;"},
                ],
                "body": {
                    "k1": ["Build task", "Changelog"],
                    "k2": ["<a href='/task/{0}/view'>{0}</a>".format(self.get_build_task_id()), changelog_template],
                },
            },
        })

        items.append(self._get_quick_footer())

        return [{"content": item} for item in items]

    def _get_quick_footer(self):

        if not utils.get_or_default(self.ctx, BuildQuickParameter):
            return "<h4> Quick </h4> <br> Turned off"

        task_link = "Not started yet..."
        status_field = "No status yet..."
        stats = {}

        if _QUICK_ACCEPTANCE_TASK in self.ctx:
            acceptance_task = channel.sandbox.get_task(self.ctx[_QUICK_ACCEPTANCE_TASK])
            task_link = "<a href='/task/{taskid}/view'>{taskid}</a>".format(taskid=acceptance_task.id)
            status_field = utils.colored_status(acceptance_task.status)
            stats = acceptance_task.ctx.get(quick_acceptance.ImagesQuickAcceptance.RESPONSES_STATS_KEY, {})

        report = {
            "<h4> Quick </h4>": [
                {
                    "TaskId": task_link,
                    "Status": status_field,
                    "Empty queries": coloring.color_diff(stats.get("empty", 0) * 100, 80),
                    "Error queries": coloring.color_diff(stats.get("error", 0) * 100, 100),
                    "Unanswer queries ": coloring.color_diff(stats.get("unanswer", 0) * 100, 100),
                    "Notfetched queries": coloring.color_diff(stats.get("notfetched", 0) * 100, 100),
                }
            ]
        }

        return report

    def _run_quick_priemka(self):
        if not utils.get_or_default(self.ctx, BuildQuickParameter):
            return None

        sub_ctx = {
            quick_acceptance.ImagesQuickAcceptance.BINARY_TASK: self.ctx[self.BUILD_TASK_KEY],
        }

        sub_task = self.create_subtask(
            task_type=quick_acceptance.ImagesQuickAcceptance.type,
            input_parameters=sub_ctx,
            description=self._subtask_description(index_type=media_settings.INDEX_QUICK),
            arch=sandboxapi.ARCH_LINUX,
            se_tag=quick_acceptance.ImagesQuickAcceptance.LIMIT_SE_TAG
        )

        return sub_task.id

    def on_execute(self):
        media_settings.ImagesSettings.USE_SAAS_QUICK = True

        priemka_task.BasePriemkaTask.on_execute(self)

        if _QUICK_ACCEPTANCE_TASK not in self.ctx:
            self.ctx[_QUICK_ACCEPTANCE_TASK] = self._run_quick_priemka()

        if _GENERATION_TASKS not in self.ctx:
            result = {}

            for base_index_type in self.__get_index_types():
                if base_index_type == media_settings.INDEX_RIM:
                    continue

                if base_index_type == media_settings.INDEX_RQ:
                    meta_index_type = media_settings.INDEX_MIDDLE_RQ
                else:
                    meta_index_type = media_settings.INDEX_MIDDLE

                result[base_index_type] = self._generate_resources(meta_index_type, base_index_type)

            self.ctx[_GENERATION_TASKS] = result

        if _PERFORMANCE_TASKS not in self.ctx:
            results = {}
            for index_type, supermind_mult, query_type, custom_ctx in self.__get_performance_tests():
                supermind_str = str(supermind_mult) if supermind_mult else 'none'
                key = ','.join((index_type, query_type, supermind_str))
                key += ',custom' if custom_ctx else ''
                results[key] = self._performance_test(index_type, query_type, supermind_mult, custom_ctx=custom_ctx)
            self.ctx[_PERFORMANCE_TASKS] = results

        self.sync_subtasks()
        self._validate_performance(_PERFORMANCE_TASKS)

    @decorators.memoize
    def _get_build_basesearch_executable(self, index_type):
        res = media_settings.ImagesSettings.basesearch_executable_resource(index_type)
        res_type = str(res)
        if res_type in self.ctx:
            return self.ctx.get(res_type)
        return self._get_build_resource(res, sandboxapi.ARCH_LINUX)

    @decorators.memoize
    def _get_build_basesearch_config(self, index_type):
        res = media_settings.ImagesSettings.basesearch_config_resource(index_type)
        res_type = str(res)
        if res_type in self.ctx:
            return self.ctx.get(res_type)
        return self._get_build_resource(res, sandboxapi.ARCH_ANY)

    def _subtask_description(self, index_type, binary=None, supermind_mult=None, query_type=None):
        description = "{}, {}".format(self.get_description(), index_type)

        if supermind_mult is not None:
            description += ', mult={}'.format(supermind_mult)

        if query_type is not None:
            description += ', queries={}'.format(query_type)

        if binary is not None:
            description += ',{}'.format(binary)

        return description

    def _subtask_compare_ctx(self, index_type, basesearch_params):
        sub_ctx = {}
        for params, age in zip(basesearch_params, ("old", "new")):
            sub_ctx.update(self._subtask_basesearch_ctx(index_type, params, age))
        return sub_ctx

    def _subtask_basesearch_ctx(self, index_type, basesearch_params, age="new"):
        if age == "new":
            basesearch_executable = self._get_build_basesearch_executable(index_type)
            basesearch_config = self._get_build_basesearch_config(index_type)
        elif age == "old":
            basesearch_executable = self._get_basesearch_executable(index_type)
            basesearch_config = self._get_basesearch_config(index_type)
        elif age == "last_released":
            basesearch_executable = self._get_basesearch_executable_last_released(index_type)
            basesearch_config = self._get_basesearch_config_last_released(index_type)

        basesearch_models = self._get_basesearch_models(index_type)

        sub_ctx = {
            basesearch_params.Binary.name: basesearch_executable,
        }

        if index_type == media_settings.INDEX_QUICK:
            sub_ctx.update({
                basesearch_params.ConfigsFolder.name: basesearch_config,
                basesearch_params.StaticData.name: media_settings.ImagesSettings.get_rtyserver_static_data(),
                basesearch_params.ShardWriterConfig.name: media_settings.ImagesSettings.get_rtyserver_sw_config(),
            })
        else:
            sub_ctx.update({
                basesearch_params.Config.name: basesearch_config,
            })

        sub_ctx.update({
            basesearch_params.Cgroup.name: json.dumps(media_settings.ImagesSettings.CGROUPS[index_type])
        })

        if basesearch_models:
            sub_ctx[basesearch_params.ArchiveModel.name] = basesearch_models

        _update_if_exists(sub_ctx, basesearch_params, "PoliteMode", False)
        _update_if_exists(sub_ctx, basesearch_params, "PatchRequestThreads", False)

        return sub_ctx

    @decorators.memoize
    def _subtask_plan_id(self, index_type, query_type):
        if index_type == media_settings.INDEX_RIM:
            testenv_attributes = media_settings.ImagesSettings.testenv_basesearch_queries_attributes(
                index_type,
                query_type
            )
            return utils.get_and_check_last_resource_with_attribute(
                resource_types.BASESEARCH_PLAN,
                testenv_attributes[0],
                testenv_attributes[1],
            ).id
        else:
            generation_task_id = self.ctx[_GENERATION_TASKS][index_type]
            return self._generated_plan_id(generation_task_id, query_type)

    @decorators.memoize
    def _subtask_database_id(self, index_type):
        generation_task_id = self.ctx[_GENERATION_TASKS][index_type]
        return self._basesearch_database_id(generation_task_id, index_type)

    @decorators.memoize
    def _subtask_meta_queries_id(self, meta_type, index_type):
        testenv_attributes = media_settings.ImagesSettings.testenv_middlesearch_queries_attributes(
            meta_type,
            index_type
        )
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
            testenv_attributes[0],
            testenv_attributes[1],
        ).id

    def _get_basesearch_attributes(self, index_type, serp_type):
        return ("priemka", "yes")

    def __get_index_types(self):
        return set(test[0] for test in self.__get_performance_tests())

    def __get_performance_tests(self):
        results = []
        if utils.get_or_default(self.ctx, BuildImgsearchParameter):
            results.extend([
                (media_settings.INDEX_MAIN, None, 'search', {}),
                (media_settings.INDEX_MAIN, None, 'factors', {}),
                (media_settings.INDEX_MAIN, None, 'snippets', {'threshold': {'shooting.rps_0.5': -3.0}}),
                (media_settings.INDEX_CBIR_MAIN, None, 'search', {'threshold': {'shooting.rps_0.5': -2.0}}),
            ])
        if utils.get_or_default(self.ctx, BuildRimdaemonParameter):
            results.append((media_settings.INDEX_RIM, None, "synthetic", {}))
        if utils.get_or_default(self.ctx, BuildRqsearchParameter):
            results.append((media_settings.INDEX_RQ, None, "all", {}))
        if utils.get_or_default(self.ctx, BuildQuickParameter):
            results.append((media_settings.INDEX_QUICK, None, 'search', {'threshold': {'shooting.rps_0.5': -1.5}}))
            results.append((media_settings.INDEX_QUICK, None, 'factors', {'threshold': {'shooting.rps_0.5': -1.5}}))
            results.append((media_settings.INDEX_QUICK, None, 'snippets', {'threshold': {'shooting.rps_0.5': -3.0}}))

        return results


def _update_if_exists(sub_ctx, params, attr, value):
    """Update dictionary if specified attribute exists in parameters"""

    if hasattr(params, attr):
        sub_ctx[getattr(params, attr).name] = value


__Task__ = ImagesPriemkaBasesearchBinary
