from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import errors

from sandbox.projects.common.search import BaseTestSuperMindTask as supermind_task
from sandbox.projects.common import dolbilka
from sandbox.projects.common import utils
from sandbox.projects.common.differ import coloring
from sandbox.projects.common.search import performance as performance_task
from sandbox.projects.common.search.basesearch import task as basesearch_task
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images.basesearch import ImagesAnalyzeBasesearchPerformance as analyze_task
from sandbox.projects.images.basesearch import ImagesAnalyzeRtyserverPerformance as rtyserver_task
from sandbox.projects.images.daemons import ImagesAnalyzeRimdaemonPerformance as rimdaemon_task


class ComparePerformanceTask:
    """
        Mixin for performance tests
    """

    def _latency_footer(self, key):
        """
            Return statistics from performance tasks
        """

        if key not in self.ctx or not self.ctx[key]:
            return [{"&nbsp;": "Calculating..."}]

        results = []
        for key, task_id in self.ctx[key].iteritems():
            task = channel.sandbox.get_task(task_id)
            data = {
                "title": key,
                "task_id": task.id,
                "task_status": task.status,
            }
            data.update(task.ctx.get("new_stats", {}).get("diff", {}))
            results.append(data)
        results.sort(key=lambda k: k['title'])

        def _format(fmt, value):
            return fmt.format(value) if value is not None else "-"

        return {
            "header": [
                {"key": "test_name",   "title": "Test"},
                {"key": "test_status",   "title": "Status"},
                {"key": "shooting.latency_0.5", "title": "Latency P50"},
                {"key": "shooting.latency_0.99", "title": "Latency P99"},
                {"key": "shooting.errors", "title": "Errors"},
                {"key": "monitoring.cpu_user", "title": "CPU P50"},
            ],
            "body": {
                "test_name": ["<a href='/task/{}/view'>{}</a>".format(v["task_id"], v["title"]) for v in results],
                "test_status": [utils.colored_status(v["task_status"]) for v in results],
                "shooting.latency_0.5": [coloring.color_diff(v.get("shooting.latency_0.5"), max_diff=1) for v in results],
                "shooting.latency_0.99": [coloring.color_diff(v.get("shooting.latency_0.99"), max_diff=1) for v in results],
                "shooting.errors": [_format("{}%", v.get("shooting.errors")) for v in results],
                "monitoring.cpu_user": [coloring.color_diff(v.get("monitoring.cpu_user"), max_diff=1) for v in results],
            },
        }

    def _performance_footer(self, key):
        """
            Return statistics from performance tasks
        """

        if key not in self.ctx or not self.ctx[key]:
            return [{"&nbsp;": "Calculating..."}]

        stats_key = performance_task.NewShootingTask.new_stats_key

        results = []
        for key, task_id in self.ctx[key].iteritems():
            task = channel.sandbox.get_task(task_id)
            data = {
                "title": key,
                "task_id": task.id,
                "task_status": task.status,
            }
            data.update(task.ctx.get(stats_key, {}).get('diff', {}))
            results.append(data)
        results.sort(key=lambda k: k['title'])

        def _format(fmt, value):
            return fmt.format(value) if value is not None else "-"

        if stats_key == performance_task.NewShootingTask.new_stats_key:
            stats_types = (
                ("shooting.rps_0.5", "RPS P50", lambda v: coloring.color_diff(v.get("shooting.rps_0.5"), max_diff=-1)),
                ("shooting.rps_stddev", "RPS stddev", lambda v: _format("{:0.2f}%", v.get("shooting.rps_stddev"))),
                ("shooting.errors", "Errors", lambda v: _format("{}%", v.get("shooting.errors")))
            )
        else:
            stats_types = (
                ("median", "Median RPS", lambda v: coloring.color_diff(v.get("median"), max_diff=-1)),
                ("stddev", "Standard deviation", lambda v: _format("{:0.2f}%", v.get("stddev"))),
                ("errors", "Errors", lambda v: _format("{}%", v.get("errors")))
            )

        header = [
            {"key": "test_name",   "title": "Test"},
            {"key": "test_status",   "title": "Status"},
        ] + [
            {"key": key, "title": title} for key, title, _ in stats_types
        ]

        body = {
            "test_name": ["<a href='/task/{}/view'>{}</a>".format(v["task_id"], v["title"]) for v in results],
            "test_status": [utils.colored_status(v["task_status"]) for v in results],
        }
        for key, title, formatter in stats_types:
            body[key] = [formatter(v) for v in results]

        return {
            "header": header,
            "body": body,
        }

    def _latency_test(self, index_type, query_type, supermind_mult):
        """
            Shooting with constant RPS
        """

        custom_ctx = {
            dolbilka.DolbilkaExecutorMode.name: 'plan',
            dolbilka.DolbilkaFixedRps.name: 0,  # use default rps from plan
        }
        return self._performance_test(index_type, query_type, supermind_mult, custom_ctx=custom_ctx)

    def _performance_test(self, index_type, *args, **kwargs):
        """
            Launch performance analysis
        """
        if index_type == media_settings.INDEX_RIM:
            return self._performance_rimdaemon_test(index_type, *args, **kwargs)
        else:
            return self._performance_basesearch_test(index_type, *args, **kwargs)

    def _performance_rimdaemon_test(self, index_type, query_type, *args, **kwargs):
        """
            Launch performance analysis (rimdaemon)
        """
        plan_id = self._subtask_plan_id(index_type, query_type)

        sub_ctx = media_settings.ImagesSettings.basesearch_shooting_parameters(index_type)
        sub_ctx.update({
            rimdaemon_task.Plan1Parameter.name: plan_id,
            rimdaemon_task.Plan2Parameter.name: plan_id,
        })
        sub_ctx.update(self._subtask_compare_ctx(
            index_type,
            (rimdaemon_task.RIMDAEMON1_PARAMS, rimdaemon_task.RIMDAEMON2_PARAMS)
        ))
        sub_task = self.create_subtask(
            task_type=rimdaemon_task.ImagesAnalyzeRimdaemonPerformance.type,
            input_parameters=sub_ctx,
            description=self._subtask_description(index_type),
            arch=sandboxapi.ARCH_LINUX,
            model='e5-2650'
        )
        return sub_task.id

    def _performance_basesearch_test(self, index_type, query_type, supermind_mult, custom_ctx={}):
        """
            Launch performance analysis (basesearch)
        """
        plan_id = self._subtask_plan_id(index_type, query_type)
        databse_id = None

        if index_type == media_settings.INDEX_QUICK:
            basesearch_params_task = rtyserver_task
            analyze_task_type = rtyserver_task.ImagesAnalyzeRtyserverPerformance.type
            databse_id = self._subtask_database_id(index_type)
        else:
            basesearch_params_task = analyze_task
            analyze_task_type = analyze_task.ImagesAnalyzeBasesearchPerformance.type

        sub_ctx = media_settings.ImagesSettings.basesearch_shooting_parameters(index_type)
        sub_ctx.update({
            analyze_task.Plan1Parameter.name: plan_id,
            analyze_task.Plan2Parameter.name: plan_id,
        })

        if supermind_mult is not None:
            sub_ctx.update({
                supermind_task.EnableSuperMindParameter.name: True,
                supermind_task.SuperMindModeParameter.name: 'mind',
                supermind_task.MultParameter.name: supermind_mult
            })
        if query_type == "snippets":
            sub_ctx.update({
                basesearch_task.RamdriveEnabledParameter.name: True,
                basesearch_task.RamdriveSizeParameter.name: media_settings.ImagesSettings.SNIPPETS_RAMDRIVE_SIZE,
                basesearch_task.RamdriveFilesParameter.name: " ".join(media_settings.ImagesSettings.SNIPPETS_RAMDRIVE_FILES),
            })

        sub_ctx.update(self._subtask_compare_ctx(
            index_type,
            (basesearch_params_task.BASESEARCH1_PARAMS, basesearch_params_task.BASESEARCH2_PARAMS)
        ))

        if databse_id:
            sub_ctx.update({
                basesearch_params_task.BASESEARCH1_PARAMS.Database.name: databse_id,
                basesearch_params_task.BASESEARCH2_PARAMS.Database.name: databse_id,
            })
        sub_ctx.update(custom_ctx)

        sub_task = self.create_subtask(
            task_type=analyze_task_type,
            input_parameters=sub_ctx,
            description=self._subtask_description(index_type, supermind_mult=supermind_mult),
            arch=sandboxapi.ARCH_LINUX,
            model='e5-2650'
        )

        return sub_task.id

    def _validate_performance(self, key):
        """
            Throws SandboxTaskFailureError on degradation or any other problems
        """

        if key not in self.ctx or not self.ctx[key]:
            raise errors.SandboxTaskFailureError("No <<{}>> in context".format(key))

        stats_key = performance_task.NewShootingTask.new_stats_key

        report = ""

        for task_key, task_id in self.ctx[key].iteritems():
            task = channel.sandbox.get_task(task_id)
            diff = task.ctx.get(stats_key, {}).get('diff', {})
            test_key = 'shooting.rps_0.5'
            if test_key not in diff:
                report += "\n* No {}.diff.{} in context of {}".format(stats_key, test_key, task_key)
                continue
            threshold = task.ctx.get('threshold', {}).get(test_key, -1.0)
            if diff[test_key] < threshold:
                report += "\n* Degradation detected in {}".format(task_key)

        if report:
            raise errors.SandboxTaskFailureError(report)
