import json

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.mediasearch.acceptance import database as search_database
from sandbox.projects.common.search.basesearch import task as basesearch_task

from sandbox.projects.images import ImagesLoadBasesearchDatabase as load_task
from sandbox.projects.images.basesearch import ImagesAnalyzeBasesearchPerformance as analyze_task
from sandbox.projects.images.resources import task as resources_task


SHARDMAP_FILE_SIZES_STATS = 'shardmap_file_sizes_stats'
_SHOOTING_TIME_LIMIT = 10 * 60   # seconds


class ImagesPriemkaBasesearchDatabase(resources_task.ImagesProductionResourcesTask,
                                      search_database.BasePriemkaDatabaseTask):
    """
        Images search database acceptance task
    """

    type = 'IMAGES_PRIEMKA_BASESEARCH_DATABASE'

    ###
    def _get_tests(self):
        if self.ctx[search_database.IndexTypeParameter.name] == "main":
            return (
                (media_settings.INDEX_MAIN, "search", None),
                (media_settings.INDEX_MAIN, "search", 0.5),
                (media_settings.INDEX_MAIN, "factors", None),
                (media_settings.INDEX_MAIN, "snippets", None),
                (media_settings.INDEX_CBIR_MAIN, "search", None),
            )
        elif self.ctx[search_database.IndexTypeParameter.name] == "garbage":
            return (
                (media_settings.INDEX_GARBAGE, "search", None),
                (media_settings.INDEX_GARBAGE, "search", 0.5),
                (media_settings.INDEX_GARBAGE, "factors", None),
                (media_settings.INDEX_GARBAGE, "snippets", None),
                (media_settings.INDEX_CBIR_GARBAGE, "search", None),
            )
        else:
            raise SandboxTaskFailureError("Unsupported index_type {}".format(self.ctx[search_database.IndexTypeParameter.name]))

    def _get_index_type_for_tier(self, tier):
        return media_settings.ImagesSettings.index_type_for_tier(tier)

    def _get_fake_shardmap(self, state):
        result = {}
        for index_type, shard_count in media_settings.ImagesSettings.FAKE_SHARDMAP_CONFIGURATIONS:
            database_prefix = media_settings.ImagesSettings.basesearch_database_prefix(index_type)
            result[index_type] = []
            for shard in range(shard_count):
                result[index_type].append('{}-{:03d}-{}'.format(database_prefix, shard, state))
        return result

    ###
    def _get_database_task(self, index_type):
        return load_task.ImagesLoadBasesearchDatabase.type

    def _get_database_args(self, index_type):
        return {
            load_task.IndexTypeParameter.name: index_type,
        }

    def _get_database_resource(self, index_type):
        return media_settings.ImagesSettings.basesearch_database_resource(index_type)

    ###
    def _get_basesearch_executable(self, index_type):
        return (self.ctx.get("basesearch_{}_executable_resource_id".format(index_type)) or
                resources_task.ImagesProductionResourcesTask._get_basesearch_executable(self, index_type))

    def _get_basesearch_config(self, index_type):
        return (self.ctx.get("basesearch_{}_config_resource_id".format(index_type)) or
                resources_task.ImagesProductionResourcesTask._get_basesearch_config(self, index_type))

    def _get_basesearch_performance_task(self, index_type):
        task_type = analyze_task.ImagesAnalyzeBasesearchPerformance.type
        basesearch_params = (
            analyze_task.BASESEARCH1_PARAMS,
            analyze_task.BASESEARCH2_PARAMS,
        )
        plan_params = (
            analyze_task.Plan1Parameter,
            analyze_task.Plan2Parameter,
        )
        return task_type, basesearch_params, plan_params

    def _get_basesearch_performance_args(self, index_type, query_type):
        sub_ctx = self._get_basesearch_shooting_parameters(index_type)

        cgroup_props = json.dumps(media_settings.ImagesSettings.CGROUPS[index_type])
        sub_ctx.update({
            analyze_task.BASESEARCH1_PARAMS.Cgroup.name: cgroup_props,
            analyze_task.BASESEARCH2_PARAMS.Cgroup.name: cgroup_props,
        })

        if query_type == "snippets":
            sub_ctx.update({
                basesearch_task.RamdriveEnabledParameter.name: True,
                basesearch_task.RamdriveSizeParameter.name: media_settings.ImagesSettings.SNIPPETS_RAMDRIVE_SIZE,
                basesearch_task.RamdriveFilesParameter.name: " ".join(media_settings.ImagesSettings.SNIPPETS_RAMDRIVE_FILES),
            })

        return sub_ctx

    def _get_basesearch_shooting_parameters(self, index_type):
        return media_settings.ImagesSettings.basesearch_shooting_parameters(index_type)

    def on_execute(self):
        search_database.BasePriemkaDatabaseTask.on_execute(self)
        self.__aggregate_shard_size_info()

    def __aggregate_shard_size_info(self):
        self.ctx[SHARDMAP_FILE_SIZES_STATS] = {}
        file_sizes_prefix_len = len(load_task.FILE_SIZES_PREFIX)

        def _merge(s, e):
            for k, v in e.iteritems():
                if k.startswith(load_task.FILE_SIZES_PREFIX):
                    k = k[file_sizes_prefix_len:]
                elif k not in (media_settings.SHARD_TOTAL_SIZE_ATTRIBUTE_NAME, media_settings.SHARD_MAPPED_SIZE_ATTRIBUTE_NAME):
                    continue
                v = int(v)
                s[k] = max(v, s[k]) if k in s else v
            return s

        for age in self._ALL_AGES:
            for index_type in self._get_index_types():
                attributes = [
                    channel.sandbox.get_resource(resource_id).attributes
                    for resource_id in self._get_shard_resource_ids(age, index_type)
                ]

                statistics = reduce(_merge, attributes, {})

                self.ctx[SHARDMAP_FILE_SIZES_STATS].setdefault(age, {})[index_type] = statistics


__Task__ = ImagesPriemkaBasesearchDatabase
