import datetime
import logging
import time

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common import errors as common_errors
from sandbox.projects.common.nanny.client import (
    StaticFileUpdate,
)


Services = {
    'robot': 'acceptance_sas_quickd_yp',
    'indexerproxy': 'saas_refresh_acceptance_images_quick_indexerproxy',
    'distributors': 'saas_refresh_acceptance_images_quick_distributors',
    'base': 'saas_refresh_acceptance_images_quick_base_sas',
}


class BaseComponent(object):
    def __init__(self, task, nanny_client, activation_timeout, services):
        self._task = task
        self._task_id = str(task.id)
        self._nanny = nanny_client
        self._timeout = int(activation_timeout)
        self.__services = services

    def _stop_service(self, service_id):
        logging.info('Stop services {}'.format(service_id))
        self._nanny.shutdown(service_id, comment='Stop service')

    def _save_info_attrs(self, service_id, info_attrs, comment):
        info_attrs["snapshot_id"] = info_attrs["_id"]
        info_attrs.pop("_id")
        info_attrs["comment"] = comment
        self._nanny.set_service_info_attrs(service_id, info_attrs)

    def _own_service(self, service_id):
        info_attrs = self._nanny.get_service_info_attrs(service_id)
        labels = [label for label in info_attrs["content"]["labels"] if label["key"] != "owner_sandbox_task"]
        labels.append({
            "key": "owner_sandbox_task",
            "value": self._task_id
        })

        info_attrs["content"]["labels"] = labels
        self._save_info_attrs(service_id, info_attrs, comment="Set sandbox task {} as owner".format(self._task_id))

    def _start_service(self, service_id):
        logging.info('Start services {}'.format(service_id))
        runtime_attrs = self._nanny.get_service_runtime_attrs(service_id)
        snapshot_id = runtime_attrs["_id"]
        self._nanny.set_snapshot_state(service_id,
                                              snapshot_id=snapshot_id,
                                              state="ACTIVE",
                                              comment='Starting testing instances',
                                              recipe="default")

    def _wait_services_for_status(self, services, status="ONLINE"):
        waitfor = time.time() + self._timeout
        while time.time() < waitfor:
            active = True
            for service_id in services:
                active &= self._nanny.get_service_current_state(service_id)\
                              .get("content", {})\
                              .get("summary", {})\
                              .get("value", "") == status
            if not active:
                time.sleep(30)
            else:
                return True

        raise common_errors.TaskFailure("Failed to set status {status} \
            to services {services} \
            in {timeout} seconds".format(services=services, status=status, timeout=self._timeout))

    def _get_alive_owners(self, services_ids):
        owners = []

        for service_id in services_ids:
            info_attrs = self._nanny.get_service_info_attrs(service_id)

            if info_attrs:
                labels = info_attrs["content"]["labels"]
                for label in labels:
                    if label["key"] == "owner_sandbox_task":
                        sandbox_task_id = label["value"]
                        task = sdk2.Task.find(id=sandbox_task_id).first()
                        if task and not (task.status in ctt.Status.Group.FINISH or task.status in ctt.Status.Group.BREAK):
                            logging.info("Task {task} is running, we have to wait it".format(task=task))
                            owners.append(task.id)
                        break

        return owners

    def _check_owner(self):
        logging.info("Check owner. Current = {}".format(self._task_id))
        tasks = []
        owners = self._get_alive_owners(self.__services)

        for owner in owners:
            if str(owner) != self._task_id:
                tasks.append(owner)

        if tasks:
            raise common_errors.TaskFailure("Another acceptance tasks {} are running, you must wait, but you don't".format(tasks))

    def own_service(self):
        for service_id in self.__services:
            self._own_service(service_id)

    def find_running_tasks(self):
        return self._get_alive_owners(self.__services)


class RobotComponent(BaseComponent):
    def __init__(self, task, nanny_client, activation_timeout, mr_config_resource_id, quick_bundle_task_id):
        super(RobotComponent, self).__init__(task, nanny_client, activation_timeout, [Services["robot"]])

        self.__mr_config = mr_config_resource_id
        self.__quick_bundle_task = quick_bundle_task_id
        self.__service = Services["robot"]

    def stop(self):
        self._check_owner()

        self._stop_service(self.__service)
        self._wait_services_for_status([self.__service], "OFFLINE")

    def start(self):
        self._check_owner()

        self._start_service(self.__service)
        self._wait_services_for_status([self.__service], "ONLINE")

    def update(self):
        self._check_owner()

        mr_config_task = str(sdk2.Resource.find(id=self.__mr_config).limit(1).first().task_id)
        self._nanny.update_service_sandbox_resources(self.__service, "IMAGES_BUILD_QUICK_PACKAGE", self.__quick_bundle_task,
                                                            comment="Update quickd bundle {} for RtRobot acceptance".format(self.__quick_bundle_task))
        self._nanny.update_service_sandbox_resources(self.__service, "GET_IMAGES_MR_INDEX_CONFIG", mr_config_task,
                                                            comment="Update index config {} for RtRobot acceptance".format(mr_config_task))


class SaasComponent(BaseComponent):
    def __init__(self, task, nanny_client, activation_timeout, minutes_in_future, file_with_drop_time):
        super(SaasComponent, self).__init__(task, nanny_client, activation_timeout, [Services["base"], Services["distributors"]])

        self.__service_base = Services["base"]
        self.__service_distrib = Services["distributors"]
        self.__minutes_in_future = minutes_in_future
        self.__drop_db_indicator_path = file_with_drop_time

    def update_drop_db_ts(self, service_id):
        self._check_owner()

        now_plus_minutes = datetime.datetime.now() + datetime.timedelta(minutes=self.__minutes_in_future)
        file_updates = {}
        file_updates[self.__drop_db_indicator_path] = StaticFileUpdate(
            local_path=self.__drop_db_indicator_path,
            content=now_plus_minutes.replace(microsecond=0).isoformat(),
        )

        self._nanny.update_service_files(service_id, static_file_updates=file_updates,
                                                comment="{} <- now + {} minutes".format(self.__drop_db_indicator_path,
                                                                                        self.__minutes_in_future))

    def drop_db(self):
        self._check_owner()

        self._stop_service(self.__service_distrib)
        self._stop_service(self.__service_base)

        self._wait_services_for_status([self.__service_distrib], "OFFLINE")
        self.update_drop_db_ts(self.__service_distrib)
        self._start_service(self.__service_distrib)

        self._wait_services_for_status([self.__service_base], "OFFLINE")
        self.update_drop_db_ts(self.__service_base)
        self._start_service(self.__service_base)

        self._wait_services_for_status([self.__service_distrib, self.__service_base], "ONLINE")
