# -*- coding: utf-8 -*-_read_resource

import logging
import os
import tempfile

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects import resource_types
from sandbox.projects.images.ImagesRatedUrls import resources as rated_urls_resources
from sandbox.sdk2.helpers import subprocess as sp


LOGGER_NAME = 'get ratedurls'
logger = logging.getLogger(LOGGER_NAME)


class GetImagesRatedUrlsNirvanaBased(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 30 * 1024
        ram = 32 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Get images rated urls'
        kill_timeout = 24 * 60 * 60

        workflow = sdk2.parameters.String('nirvana workflow to clone',
                                           default='',
                                           required=False)

        timestamp = sdk2.parameters.String('timestamp (YYYY-MM-DDThh:mm:ss+0300)',
                                           default='',
                                           required=False)

        rated_urls_tool = sdk2.parameters.LastReleasedResource('ratedurls tool binary',
                                                               resource_type=rated_urls_resources.IMAGES_RATED_URLS_EXECUTABLE,
                                                               state=(ctr.State.READY, ),
                                                               required=True)

        adult_rated_urls = sdk2.parameters.LastReleasedResource('adult rated urls list',
                                                                resource_type=resource_types.IMAGES_ADULT_RATED_URLS,
                                                                state=(ctr.State.READY, ),
                                                                required=True)

        rated_convert_tool = sdk2.parameters.LastReleasedResource('rated urls convert to hashes tool binary',
                                                                  resource_type=resource_types.IMAGES_MR_INDEX_RATEDCONVERT,
                                                                  state=(ctr.State.READY, ),
                                                                  required=True)

    def on_execute(self):
        rated_urls_tool_path = str(sdk2.ResourceData(self.Parameters.rated_urls_tool).path)
        temp_file = tempfile.NamedTemporaryFile()

        nirvana_oauth = sdk2.Vault.data('IMAGES-ROBOT', 'robot-images-sandbox-nirvana-token')
        yt_oauth = sdk2.Vault.data('IMAGES-ROBOT', 'yt_token')

        logger.info('Start preparing rated urls')
        env = os.environ.copy()
        env['LC_ALL'] = 'C'

        workflow_param = [] if not self.Parameters.workflow else ['--workflow', self.Parameters.workflow]
        ts_param = [] if not self.Parameters.timestamp else ['--timestamp', self.Parameters.timestamp]

        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            proc_pipes = {'stdout': pl.stdout, 'stderr': pl.stderr}
            sp.check_call([rated_urls_tool_path + '/ratedurls',
                           '--nirvana-oauth',
                           nirvana_oauth,
                           '--yt-oauth',
                           yt_oauth,
                           '--output',
                           temp_file.name] + workflow_param + ts_param,
                          env=env,
                          **proc_pipes)

            logger.info('Number of adult rated urls:')
            adult_rated_urls_path = str(sdk2.ResourceData(self.Parameters.adult_rated_urls).path)
            sp.check_call(['wc', '-l', adult_rated_urls_path], env=env, **proc_pipes)

            logger.info('Merging adult rated urls with relevance rated urls')
            rated_urls_resource = resource_types.IMAGES_RATED_URLS(self, 'images  rated urls', 'ratedurls.txt')
            rated_urls_resource_data = sdk2.ResourceData(rated_urls_resource)
            rated_urls_resource_data_path = str(rated_urls_resource_data.path)

            sp.check_call(['sort', '-u', '-t', '\t', '-S', '1G',
                           '-o', rated_urls_resource_data_path,
                           adult_rated_urls_path, temp_file.name],
                          env=env,
                          **proc_pipes)

            rated_urls_resource_data.ready()

            logger.info('Final number of rated urls:')
            sp.check_call(['wc', '-l', rated_urls_resource_data_path], env=env, **proc_pipes)

            rated_convert_path = str(sdk2.ResourceData(self.Parameters.rated_convert_tool).path)

            hashes_resource = resource_types.IMAGES_RATED_URL_HASHES(self, 'images rated url hashes', 'ratedurlshashes.bin')
            hashes_resource_data = sdk2.ResourceData(hashes_resource)
            hashes_resource_data_path = str(hashes_resource_data.path)

            logger.info('Convert URLs to hashes')
            sp.check_call([rated_convert_path,
                           '--input-file', rated_urls_resource_data_path,
                           '--output-file', hashes_resource_data_path],
                          env=env,
                          **proc_pipes)

            hashes_resource_data.ready()
