#  -*- coding: utf-8 -*-

from datetime import datetime

import hashlib
import logging
import pytz

import sandbox.projects.ReportDataRuntimeItem as RDRI
import sandbox.projects.ReportDataRuntimeTags as RDRT

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.resource import State
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import IMGS_REPORT_DATA_RUNTIME_BUNDLE


IMAGES_PROJECT_NAME = 'IMGS'
NANNY_API_URL = 'http://nanny.yandex-team.ru/'
NANNY_REPORT_DATA_RUNTIME_LOCAL_NAME = 'data.runtime'
NANNY_REPORT_SERVICE_NAME = 'production_report_sas_imgs_yp'

SECONDS_IN_HOUR = 3600
TIMEZONE = pytz.timezone('Europe/Moscow')


def get_logo_file_path(base_path, resource_id):
    version_path = None
    for version in base_path.iterdir():
        if (version / '.metainfo/sandbox.json').is_file():
            version_path = version
            break
    if not version_path:
        raise TaskFailure("Cannot find data for resource {} at {}".format(resource_id, base_path))

    logo_path = version_path / 'data/logo/desktop.json'
    if not logo_path.is_file():
        raise TaskFailure("Cannot find logo file for resource {} at {}".format(resource_id, str(logo_path)))
    logging.info("Found logo for resource {} at {}".format(resource_id, logo_path))
    return logo_path


def calc_logo_md5(bundle):
    logo_path = get_logo_file_path(sdk2.ResourceData(bundle).path, bundle.id)
    with open(str(logo_path), 'rb') as fin:
        return hashlib.md5(fin.read()).hexdigest()


class ImagesReleaseReportDataRuntimeBundle(sdk2.Task):
    """
    Релизит последнюю собранную версию data.runtime, если выполнены критерии:
    * последний построенный бандл создан позже, чем последний зарелиженный
    * либо после релиза прошло >= 1 дня, либо md5 файлов с лого отличаются
    """

    class Parameters(sdk2.Task.Parameters):
        nanny_oauth_token_vault_key = sdk2.parameters.YavSecret("Nanny OAuth token vault key", required=True)

        bundle_ttl = sdk2.parameters.Integer("Released resource ttl in days", default=365, required=False)
        release_cooldown = sdk2.parameters.Integer("Hours to wait until next release after last resource was released in SB", default=6, required=False)
        nanny_cooldown = sdk2.parameters.Integer("Hours to wait until next release after last resource in Nanny was released", default=24, required=False)

        release_on_weekends = sdk2.parameters.Bool("Release resource on weekends", default=False, required=False)
        release_hours_start = sdk2.parameters.Integer("Start hour of the release time interval", default=10, required=False)
        release_hours_duration = sdk2.parameters.Integer("Duration of the release time interval (in hours)", default=13, required=False)

        release_type = sdk2.parameters.String(
            'Resource release type',
            ui=sdk2.parameters.String.UI('select'),
            choices=[('stable', ReleaseStatus.STABLE), ('unstable', ReleaseStatus.UNSTABLE), ('testing', ReleaseStatus.TESTING)],
            default='testing')

    class Requirements(sdk2.Requirements):
        disk_space = 100
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        # get current nanny resource
        nanny_bundle_id = self.__get_resource_id_from_nanny()
        nanny_bundle = self.__find_bundle({'id': nanny_bundle_id},
                                          found_format_msg="Found nanny resource {}",
                                          err_msg="Cannot find nanny resource {} with id {}".format(IMGS_REPORT_DATA_RUNTIME_BUNDLE.type, nanny_bundle_id))

        rdrt = RDRT.ReportDataRuntimeTags()
        rdrt.ctx = {RDRI.Project.name : IMAGES_PROJECT_NAME}

        # get last built resource
        last_built_bundle = self.__find_bundle({'resource_type': rdrt.get_data_resource_type_by_project(), 'state': State.READY},
                                               found_format_msg="Found last built resource {}",
                                               err_msg="Cannot find last built resource {}".format(IMGS_REPORT_DATA_RUNTIME_BUNDLE.type))

        # get last created released resource
        last_released_bundle = self.__find_bundle({'resource_type': rdrt.get_data_resource_type_by_project(), 'state': State.READY, 'attrs': dict(released=ReleaseStatus.STABLE)},
                                                  found_format_msg="Found last released resource {}",
                                                  err_msg="Cannot find last released resource {}".format(IMGS_REPORT_DATA_RUNTIME_BUNDLE.type))

        if self.__need_release(nanny_bundle, last_built_bundle, last_released_bundle):
            release_params={'task_id' : last_built_bundle.task_id,
                            'type' : self.Parameters.release_type,
                            'subject': "data.runtime for project {} ({})".format(IMAGES_PROJECT_NAME, self.Parameters.release_type),
                            'params': {'ttl': self.Parameters.bundle_ttl}}
            self.server.release(release_params)

            logging.info("Released resource with id {} to {} with ttl = {} days".format(last_built_bundle.id, self.Parameters.release_type, last_built_bundle.ttl))

    def __find_bundle(self, find_params, found_format_msg="Found resource id={}", err_msg=None):
        if not err_msg:
            err_msg = "Cannot find resource by params {}".format(str(find_params))

        resource = sdk2.Resource.find(**find_params).first()
        if not resource:
            raise TaskFailure(err_msg)

        logging.info(found_format_msg.format(resource.id))
        return resource

    def __get_resource_id_from_nanny(self):
        nanny_client = nanny.NannyClient(api_url=NANNY_API_URL, oauth_token=self.Parameters.nanny_oauth_token_vault_key.data())
        resources = nanny_client.get_service_resources(NANNY_REPORT_SERVICE_NAME)
        if not isinstance(resources, dict) or 'content' not in resources.keys() or \
           not isinstance(resources['content'], dict) or 'sandbox_files' not in resources['content'].keys():
            raise TaskFailure("Cannot access sandbox files in resources of nanny service {}".format(NANNY_REPORT_SERVICE_NAME))

        nanny_bundle_id = None
        for resource_info in resources['content']['sandbox_files']:
            if resource_info['local_path'] == NANNY_REPORT_DATA_RUNTIME_LOCAL_NAME:
                nanny_bundle_id = resource_info['resource_id']
                break
        if not nanny_bundle_id:
            raise TaskFailure("Cannot find resource {} of nanny service {}".format(NANNY_REPORT_DATA_RUNTIME_LOCAL_NAME, NANNY_REPORT_SERVICE_NAME))

        return nanny_bundle_id

    class BundleInfo(object):
        def __init__(self, bundle, logo_md5):
            self.bundle = bundle
            self.logo_md5 = logo_md5

    def __need_release(self, nanny_bundle, last_built_bundle, last_released_bundle):
        nanny_bundle_info = self.BundleInfo(nanny_bundle, calc_logo_md5(nanny_bundle))
        last_built_bundle_info = self.BundleInfo(last_built_bundle, calc_logo_md5(last_built_bundle))
        last_released_bundle_info = self.BundleInfo(last_released_bundle, calc_logo_md5(last_released_bundle))

        return self.__is_time_appropriate() and \
               last_built_bundle.created > nanny_bundle.created and \
               self.__are_different_resources(last_built_bundle_info, nanny_bundle_info, self.Parameters.nanny_cooldown) and \
               last_built_bundle.created > last_released_bundle.created and \
               self.__are_different_resources(last_built_bundle_info, last_released_bundle_info, self.Parameters.release_cooldown)

    def __is_time_appropriate(self):
        current_time = datetime.now(tz=TIMEZONE)
        appropriate_day = True if self.Parameters.release_on_weekends else current_time.weekday() < 5
        appropriate_hours = self.Parameters.release_hours_start <= current_time.hour and \
                            current_time.hour < self.Parameters.release_hours_start + self.Parameters.release_hours_duration
        logging.info("Appropriate day: {}, appropriate hours: {}".format(appropriate_day, appropriate_hours))
        return appropriate_day and appropriate_hours

    def __is_day_time(self):
        current_time = datetime.now(tz=TIMEZONE)
        return 9 < current_time.hour < 22

    def __need_update_resource(self, first_bundle_info, second_bundle_info, cooldown):
        return (first_bundle_info.bundle.created - second_bundle_info.bundle.created).total_seconds() >= cooldown * SECONDS_IN_HOUR and self.__is_day_time()

    def __are_different_resources(self, first_bundle_info, second_bundle_info, cooldown):
        are_different = self.__need_update_resource(first_bundle_info, second_bundle_info, cooldown) or first_bundle_info.logo_md5 != second_bundle_info.logo_md5
        logging.info("Resources {} and {} are different: {}".format(first_bundle_info.bundle.id, second_bundle_info.bundle.id, are_different))
        return are_different
