# -*- coding: utf-8 -*-

import logging
import time
import os
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters

import sandbox.projects.resource_types as resource_types
import sandbox.projects.common.mapreduce_runner as mapreduce_runner
import sandbox.projects.common.mapreduce_stored_tables as mrst

from sandbox.common import errors


class MrServerParameter(sdk_parameters.SandboxStringParameter):
    name = 'mr_server'
    description = 'Mapreduce server ("local" in a locally set up MR microcluster):'
    default_value = 'local'
    required = True


class MapreduceExecutableResourceParameter(sdk_parameters.LastReleasedResource):
    name = 'mapreduce_resid'
    description = 'mapreduce-executable resource'
    resource_type = resource_types.MAPREDUCE_EXECUTABLE


class YaMakeProgramParameter(sdk_parameters.ResourceSelector):
    name = 'yamake_output_resid'
    description = 'YA_MAKE task with binary'
    resource_type = resource_types.BUILD_OUTPUT


class YaMakeFallbackProgramParameter(sdk_parameters.ResourceSelector):
    name = 'yamake_output_fallback_resid'
    description = 'YA_MAKE task with binary in case of first YA_MAKE failure'
    resource_type = resource_types.BUILD_OUTPUT


class YaMakeProgramPathParameter(sdk_parameters.SandboxStringParameter):
    name = 'yamake_program_path'
    description = 'binary path in BUILD_OUTPUT of YA_MAKE task.'


class ProgramParameter(sdk_parameters.ResourceSelector):
    name = 'program_resid'
    description = 'program resource'
    resource_type = resource_types.IMAGES_MR_INDEX_METADOC


class ProgramFallbackParameter(sdk_parameters.ResourceSelector):
    name = 'program_fallback_resid'
    description = 'program resource used if no program_resid available'
    resource_type = resource_types.IMAGES_MR_INDEX_METADOC


class SystemCommand(sdk_parameters.SandboxStringParameter):
    name = 'syscommand'
    description = 'system command to do before ordinary command'
    multiline = True
    default_value = ''


class Command(sdk_parameters.SandboxStringParameter):
    name = 'command'
    description = 'command'
    multiline = True
    default_value = ''


class Input1TablesParameter(sdk_parameters.ResourceSelector):
    name = 'input_tables1'
    description = 'input tables'
    resource_type = resource_types.USERDATA_TABLES_ARCHIVE


class Input2TablesParameter(sdk_parameters.ResourceSelector):
    name = 'input_tables2'
    description = 'input tables'
    resource_type = resource_types.USERDATA_TABLES_ARCHIVE


class Input3TablesParameter(sdk_parameters.ResourceSelector):
    name = 'input_tables3'
    description = 'input tables'
    resource_type = resource_types.USERDATA_TABLES_ARCHIVE


class Input4TablesParameter(sdk_parameters.ResourceSelector):
    name = 'input_tables4'
    description = 'input tables'
    resource_type = resource_types.USERDATA_TABLES_ARCHIVE


class Input5TablesParameter(sdk_parameters.ResourceSelector):
    name = 'input_tables5'
    description = 'input tables'
    resource_type = resource_types.USERDATA_TABLES_ARCHIVE


class ConfigParamater(sdk_parameters.ResourceSelector):
    name = 'config_resid'
    description = 'config directory or other files'
    resource_type = resource_types.IMAGES_MR_INDEX_CONFIG


class ResultTableListParameter(sdk_parameters.SandboxStringParameter):
    name = 'result_tables'
    description = 'List of result mr-tables'
    multiline = True


class ResultResourceDescriptionParameter(sdk_parameters.SandboxStringParameter):
    name = 'resource_description'
    description = 'Resource name for output tables'


class ImagesTestMrindex(sdk_task.SandboxTask):
    """
        Таск для тестирования чего бы то ни было
    """
    type = 'IMAGES_TEST_MRINDEX'
    input_parameters = [
        MrServerParameter,
        MapreduceExecutableResourceParameter,
        ProgramParameter,
        ProgramFallbackParameter,
        YaMakeProgramParameter,
        YaMakeProgramPathParameter,
        SystemCommand,
        Command,
        Input1TablesParameter,
        Input2TablesParameter,
        Input3TablesParameter,
        Input4TablesParameter,
        Input5TablesParameter,
        ConfigParamater,
        ResultTableListParameter,
        ResultResourceDescriptionParameter,
    ]
    server_runner = None

    def on_execute(self):
        try:
            self.setup_server_if_needed()
            self.run()
        finally:
            self.teardown_server()

    def sync_archive_from_param(self, name):
        if not self.ctx[name]:
            return
        if self.ctx[name] == 'null':  # happens when copying tasks
            return

        self.mr_syncer.resource2tables(self, int(self.ctx[name]))

    def run(self):

        if self.ctx[YaMakeProgramPathParameter.name]:
            primary_binary_ok = False
            try:
                build_output = self.sync_resource(self.ctx[YaMakeProgramParameter.name])
                self.metadoc_bin = build_output + self.ctx[YaMakeProgramPathParameter.name]
                primary_binary_ok = True
            except errors.TaskError:
                pass
            if not primary_binary_ok:
                build_output = self.sync_resource(self.ctx[YaMakeProgramParameter.name])
                self.metadoc_bin = build_output + self.ctx[YaMakeProgramPathParameter.name]
        else:
            try:
                self.metadoc_bin = self.sync_resource(self.ctx[ProgramParameter.name])
            except errors.TaskError:
                self.metadoc_bin = self.sync_resource(self.ctx[ProgramFallbackParameter.name])

        if self.ctx[ConfigParamater.name]:
            self.config_dir = self.sync_resource(self.ctx[ConfigParamater.name])
        else:
            self.config_dir = '/no/config'
        self.mr_syncer = mrst.MapreduceStoredTables(self.mapreduce_bin, self.mapreduce_server)

        self.sync_archive_from_param(Input1TablesParameter.name)
        self.sync_archive_from_param(Input2TablesParameter.name)
        self.sync_archive_from_param(Input3TablesParameter.name)
        self.sync_archive_from_param(Input4TablesParameter.name)
        self.sync_archive_from_param(Input5TablesParameter.name)

        syscmd = self.ctx[SystemCommand.name]
        if syscmd:
            logging.info("will run %s", syscmd.format(metadoc_bin=self.metadoc_bin,
                                                      mapreduce_bin=self.mapreduce_bin,
                                                      program=self.metadoc_bin,
                                                      config_dir=self.config_dir,
                                                      mr_server=self.mapreduce_server))
            sdk_process.run_process(
                "set -x\n" + syscmd.format(metadoc_bin=self.metadoc_bin,
                                           program=self.metadoc_bin,
                                           config_dir=self.config_dir,
                                           mr_server=self.mapreduce_server),
                log_prefix='system_cmd', shell=True)

        cmd = self.ctx[Command.name]
        cmd = cmd.format(metadoc_bin=self.metadoc_bin,
                         program=self.metadoc_bin,
                         config_dir=self.config_dir,
                         mr_server=self.mapreduce_server)
        logging.info("will run %s", cmd)
        start_time = time.time()

        environment = os.environ.copy()
        environment['MR_OPT'] = 'stderrlevel=1'
        sdk_process.run_process(cmd.split(), log_prefix='program', environment=environment)

        finish_time = time.time()
        self.ctx['mr_cmd_duration'] = finish_time - start_time

        tables = [t for t in self.ctx[ResultTableListParameter.name].split()]

        for t in tables:
            sdk_process.run_process(
                '''{mapreduce_bin} -server {mr_server} -sort {table}'''.format(
                    mapreduce_bin=self.mapreduce_bin,
                    table=t,
                    mr_server=self.mapreduce_server,
                ), log_prefix='mrsort', shell=True)

        attrs = {'needed_for_tests': self.ctx.get('needed_for_tests', ''),
                 'testenv_database': self.ctx.get('testenv_database', ''),
                 'testenv_test_name': self.ctx.get('testenv_test_name', ''),
                 'rev': self.ctx.get('rev', ''),
                 }
        auto_update_tag = self.ctx.get('out_res_auto_update_tag', '')
        if auto_update_tag:
            attrs['autoupdate_' + auto_update_tag] = 'yes'

        list_cmd = ('''
            echo "placeholder\t" | {mapreduce} -server {mr_server} -write sandbox/full_table_list
            {mapreduce} -server {mr_server} -list -prefix sandbox/ | \
            sed "s,^,\t,g" | \
            {mapreduce} -server {mr_server} -write sandbox/full_table_list''')
        list_cmd = list_cmd.format(mapreduce=self.mapreduce_bin,
                                   program=self.metadoc_bin,
                                   config_dir=self.config_dir,
                                   mr_server=self.mapreduce_server)
        logging.info("will run %s", list_cmd)
        sdk_process.run_process(list_cmd, log_prefix='mr_list', shell=True)

        self.mr_syncer.tables2resource(
            self,
            self.ctx['resource_description'],
            prefix='sandbox/',
            flt=lambda x: x in tables or x == 'sandbox/full_table_list',
            attrs=attrs
        )

    def setup_server_if_needed(self):
        self.mapreduce_server = self.ctx[MrServerParameter.name]
        self.mapreduce_bin = self.sync_resource(self.ctx[MapreduceExecutableResourceParameter.name])

        if self.mapreduce_server == 'local':
            self.server_runner = mapreduce_runner.MapreduceRunner(self.mapreduce_bin,
                                                                  self.abs_path('mr_dir'),
                                                                  self.client_info)
            self.server_runner.setup_server()
            self.mapreduce_server = self.server_runner.mr_server_string
            logging.info("Set up local server on %s" % self.mapreduce_server)

    def teardown_server(self):
        if self.server_runner is not None:
            self.server_runner.teardown_server()


__Task__ = ImagesTestMrindex
