import os
import time
import logging
from sandbox import sandboxsdk
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

import sandbox.projects.images.basesearch.resources as images_resources
import sandbox.projects.images.models.resources as images_models_resources

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search.components import FusionSearch
from sandbox.projects.common.nanny.nanny import NannyClient

from sandbox.sandboxsdk.channel import channel


class RunFusionProductionEnvBase(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 61440
        client_tags = (
            ctc.Tag.LXC
        )

    class Parameters(sdk2.Task.Parameters):
        nanny_service_name = sdk2.parameters.String(
            'Nanny service name',
            required=True,
            default_value="saas_refresh_production_images_base",
        )

        docs_to_index = sdk2.parameters.Integer(
            'Document to index from distributors',
            required=True,
            default_value=1000,
        )

        rtyserver_binary = sdk2.parameters.Resource(
            'RTYServer executable',
            resource_type=resource_types.IMGSEARCH_RTYSERVER_EXECUTABLE,
            state=(ctr.State.READY, ),
            required=False,
        )

        rtyserver_configs_bundle = sdk2.parameters.Resource(
            'rtyserver_configs_bundle',
            resource_type=images_resources.IMGULTRA_SAAS_RTYSERVER_CONFIGS_BUNDLE,
            state=(ctr.State.READY, ),
            required=False,
        )

        models_archive = sdk2.parameters.Resource(
            'models.archive',
            resource_type=images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE,
            state=(ctr.State.READY, ),
            required=False,
        )

        models = sdk2.parameters.Resource(
            'models',
            resource_type=resource_types.RTYSERVER_MODELS,
            state=(ctr.State.READY, ),
            required=False,
        )

    def __get_resource_from_nanny(self, nanny_resources_descr, resource_type):
        for sb_file in nanny_resources_descr['sandbox_files']:
            if sb_file['resource_type'] == resource_type:
                if 'resource_id' in sb_file:
                    return channel.task.sync_resource(sb_file['resource_id'])
                elif 'task_id' in sb_file:
                    resource = sdk2.Resource[resource_type].find(task=sdk2.Task[int(sb_file['task_id'])]).first()
                    return str(sdk2.ResourceData(resource).path)
        return None

    def __get_static_files(self, static_section):
        res = {}
        for item in static_section:
            local_path = item["local_path"]
            res[local_path] = item["content"]
        return res

    def make_local_fixes(self):
        create_symlink(self.obtained_resources['models_archive'], './models.archive')
        self.obtained_resources['models_archive'] = './models.archive'

    def get_extra_params(self):
        detach_path = sandboxsdk.paths.make_folder("detach")
        state_path = sandboxsdk.paths.make_folder("state")
        rtindex_path = sandboxsdk.paths.make_folder("shm")
        current_path = os.getcwd()
        extra_params = {
                "CONFIG_PATH": os.path.join(current_path, self.config_bundle_dir),
                "MODELS_PATH": current_path + "/",
                "DISTRIBUTORS_PATH": current_path,
                "DETACH_DIRECTORY": detach_path,
                "STATE_ROOT": state_path,
                "RTINDEX_DIRECTORY": rtindex_path,
                "BSCONFIG_INAME": "refresh",
                "CPU_LIMIT": "8",
                "CTYPE": "integrations_tests",
        }
        return extra_params

    @classmethod
    def __process_params(cls):
        logging.info("Processing params for \"%s\"", str(cls))

        result = {}

        for member in cls.Parameters:
            if not issubclass(member, sdk2.parameters.Resource):
                logging.info('Skipping attribute \"%s\" since it is not a subclass/class of sdk2.parameters.Resource', str(member))
                continue

            eh.ensure(not(member.name in result.keys()), "Several resources with same name \"{}\" in \"{}\"".format(member.name, str(cls)))
            result[member.name] = member.resource_type

        logging.info("Processed params, dict of required resources: \"%s\"", str(result))

        return result

    def on_execute(self):
        nanny_token = sdk2.Vault.data('MEDIA_DEPLOY', 'nanny-oauth-token')
        nanny_service = self.Parameters.nanny_service_name
        nanny_url = "https://nanny.yandex-team.ru/"
        nanny_client = NannyClient(nanny_url, nanny_token)
        nanny_attrs = nanny_client.get_service_runtime_attrs(nanny_service)
        nanny_resources_descr = nanny_attrs["content"]["resources"]

        tvm_secret = sdk2.Vault.data('MEDIA_DEPLOY', 'ultra-test-tvm-secret')
        tvm_client_id = sdk2.Vault.data('MEDIA_DEPLOY', 'ultra-test-tvm-client-id')

        os.environ["TVM_SECRET"] = tvm_secret
        os.environ["TVM_CLIENT_ID"] = tvm_client_id

        required_resources = self.__process_params()
        self.obtained_resources = {}

        for resource_name, resource_type in required_resources.iteritems():
            value = getattr(self.Parameters, resource_name)
            if value:
                resource_path = str(sdk2.ResourceData(value).path)
                logging.info('Resource \"%s\" with resource type \"%s\" is under \"%s\" and was taken from parameters', resource_name, str(resource_type), resource_path)
            else:
                resource_path = self.__get_resource_from_nanny(nanny_resources_descr, resource_type)
                logging.info('Resource \"%s\" with resource type \"%s\" is under \"%s\" and was taken from nanny service', resource_name, str(resource_type), resource_path)

            eh.ensure(resource_path, "Failed to obtain \"{}\" with type \"{}\"".format(resource_name, str(resource_type)))
            basename = os.path.basename(resource_path)
            create_symlink(resource_path, os.path.join("./", basename))
            if resource_name == 'rtyserver_configs_bundle':
                self.config_bundle_dir = basename
                self.obtained_resources['fusion_search_config'] = os.path.join(basename, 'rtyserver.conf-common')
                self.obtained_resources['base_search_config'] = os.path.join(basename, 'basesearch-refresh')
                self.obtained_resources['environment'] = os.path.join(basename, 'environment')
            else:
                self.obtained_resources[resource_name] = basename

        store_local_files(self.__get_static_files(nanny_resources_descr.get("static_files", {})))

        workdir_path = sandboxsdk.paths.make_folder("workdir")

        logging.info("Sandbox resources obtained")

        self.make_local_fixes()
        logging.info("Local fixes accomplished")

        # create empty file, production file generated by jinja
        templates_path = sandboxsdk.paths.make_folder("templates")

        # can't find it in our spec
        open(templates_path + "/replicas.conf", "w").close()
        open("replicas.conf", "w").close()  # TODO(yrum): remove this after 2018-07-31 (when replicas.conf is made by Nanny)

        fusion_search = FusionSearch(workdir_path, None,
                                     fusion_config_path=self.obtained_resources['fusion_search_config'],
                                     config_file=self.obtained_resources['base_search_config'],
                                     environment_file_path=self.obtained_resources['environment'],
                                     binary=self.obtained_resources['rtyserver_binary'],
                                     static_data_path=self.obtained_resources['models'],
                                     extra_params=self.get_extra_params(),
                                     from_sdk2=True,
                                     )
        logging.info("Fusion config: %s", fusion_search.fusion_config_path)

        fusion_search.start()
        fusion_search.wait(timeout=3600)  # wait for DB backup to be fetched
        while True:
            result = fusion_search.get_info_server()['result']
            docs_in_disk = int(result['docs_in_disk_indexers'])
            final_indices_count = count_final_indices(result)
            logging.info("Fusion has: %u docs in disk index and %u final indices", docs_in_disk, final_indices_count)
            if docs_in_disk > self.Parameters.docs_to_index:
                break
            time.sleep(30)
        fusion_search.run_fusion_command("pause_docfetcher")

        time.sleep(30)  # 10s may be not enough to empty indexing queues
        fusion_search.run_fusion_command("wait_empty_indexing_queues")
        fusion_search.run_fusion_command("reopen_indexes")
        time.sleep(30)  # 10s may be not enough to close indexers
        fusion_search.run_fusion_command("create_merger_tasks")

        while True:
            result = fusion_search.get_info_server()['result']
            docs_in_disk = int(result['docs_in_disk_indexers'])
            final_indices_count = count_final_indices(result)
            logging.info("Fusion merging has: %u docs in disk index and %u final indices", docs_in_disk, final_indices_count)
            if docs_in_disk == 0 and final_indices_count == 1:
                break
            time.sleep(30)
        fusion_search.kill_softly()


def count_final_indices(info_server_result):
    indices = info_server_result['indexes']
    return sum(index['type'] == 'FINAL' and index['realm'] == 'Persistent' for _, index in indices.iteritems())


def store_local_files(local_files):
    for fname, fcontent in local_files.iteritems():
        fu.write_file(fname, fcontent)


def create_symlink(resource, dest_name):
    if not os.path.exists(dest_name):
        os.symlink(str(resource), dest_name)
        logging.info("Symlinked \"%s\"->\"%s\"", str(resource), dest_name)
