# -*- coding: utf-8 -*-

import time
import json

from sandbox.projects import GetImagesMrIndexConfig as mr_index_config
from sandbox.projects.images import ImagesBuildSearchBinary as build_base
from sandbox.projects.images.basesearch import resources as images_basesearch_resources
from sandbox.projects.images.models import ImagesBuildDynamicModels as build_models
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.common.types.task import Status
from sandbox import sdk2

import logging
import os


class RemoteNannySaasComponent:
    def __init__(self,
                 task_id,
                 nanny_client,
                 gencfg,
                 build_base_task,
                 build_models_task,
                 mr_index_config_task,
                 production_services,
                 testing_services,
                 gencfg_groups,
                 config_files,
                 nanny_category,
                 clear_cache_script,
                 activation_timeout):
        self.__nanny_client = nanny_client
        self.__gencfg = gencfg
        self.__build_base_task = str(build_base_task) if build_base_task else ''
        self.__build_models_task = str(build_models_task) if build_models_task else ''
        self.__mr_index_config_task = str(mr_index_config_task) if mr_index_config_task else ''
        self.__production_services = production_services
        self.__testing_services = testing_services
        self.__gencfg_groups = gencfg_groups
        self.__config_files = config_files
        self.__nanny_category = str(nanny_category)
        self.__clear_cache_script = str(clear_cache_script)
        self.__activation_timeout = int(activation_timeout)
        self.__task_id = str(task_id)

    def __remove_tickets_integration_rules(self, service_id):
        info_attrs = self.__nanny_client.get_service_info_attrs(service_id)
        tickets_integration = info_attrs["content"]["tickets_integration"]
        tickets_integration["service_release_tickets_enabled"] = False
        tickets_integration["service_release_rules"] = []
        self.__save_info_attrs(service_id, info_attrs, comment="tickets integration were removed")

    def __save_info_attrs(self, service_id, info_attrs, comment):
        info_attrs["snapshot_id"] = info_attrs["_id"]
        info_attrs.pop("_id")
        info_attrs["comment"] = comment
        self.__nanny_client.set_service_info_attrs(service_id, info_attrs)

    def __set_media_recipes(self,
                            service_id,
                            stop_degrade_level=0.4,
                            prepare_degrade_level=1.0,
                            operating_degrade_level=1.0):
        info_attrs = self.__nanny_client.get_service_info_attrs(service_id)
        info_attrs["content"]["recipes"]["content"] = [
            {
                "desc": "Common deploy script",
                "context": [
                    {
                        "value": "sas",
                        "key": "geo"
                    },
                    {
                        "key": "prepare_degrade_level",
                        "value": str(prepare_degrade_level)
                    },
                    {
                        "key": "operating_degrade_level",
                        "value": str(operating_degrade_level)
                    },
                    {
                        "key": "stop_degrade_level",
                        "value": str(stop_degrade_level)
                    }
                ],
                "name": "deploy_saas_by_loc.yaml",
                "id": "common"
            }
        ]

        self.__save_info_attrs(service_id, info_attrs, comment="media recipes were set")

    def __set_base_search_task(self, service_id, task_id):
        self.__nanny_client.update_service_sandbox_file(
            service_id,
            build_base.ImagesBuildSearchBinary.type,
            task_id,
            comment="Changed by task {}".format(self.__task_id),
            allow_empty_changes=True
        )

    def __set_dynamic_models_task(self, service_id, task_id):
        self.__nanny_client.update_service_sandbox_file(
            service_id,
            build_models.ImagesBuildDynamicModels.type,
            task_id,
            comment="Changed by task {}".format(self.__task_id),
            allow_empty_changes=True
        )

    def __set_mrindex_config_task(self, service_id, task_id):
        self.__nanny_client.update_service_sandbox_file(
            service_id,
            mr_index_config.GetImagesMrIndexConfig.type,
            task_id,
            comment="Changed by task {}".format(self.__task_id),
            allow_empty_changes=True
        )

    def __get_cms_last_tag_url(self):
        tag = self.__gencfg.get_last_stable_tag('unstable')
        assert tag is not None
        return 'https://cmsearch.n.yandex.ru/res/gencfg/' + tag + "/w-generated/all/"

    def __set_config_files(self, service_id, config_files):
        resources = self.__nanny_client.get_service_resources(service_id)
        content = resources['content']

        for url_file in content['url_files']:
            if url_file['local_path'] in config_files:
                url_file['url'] = self.__get_cms_last_tag_url() + config_files[url_file['local_path']]
        resources['content'] = content
        resources["comment"] = "test config files were set"
        self.__nanny_client.update_service_resources(service_id, resources)

    def __set_ssd_clear_cache_hook(self, service_id, clear_cache_script):
        ISS_HOOK_FILENAME = 'iss_hook_uninstall'
        resources = self.__nanny_client.get_service_resources(service_id)
        content = resources['content']
        script_changed = False
        for url_file in content['url_files']:
            if url_file['local_path'] == ISS_HOOK_FILENAME:
                url_file['url'] = clear_cache_script
                script_changed = True
        if not script_changed:
            content['url_files'].append({'local_path': ISS_HOOK_FILENAME,
                                         'url': clear_cache_script,
                                         'is_dynamic': False})

        resources['content'] = content
        resources["comment"] = "uninstall_script was set, test sandbox task: {}".format(self.__task_id)
        self.__nanny_client.update_service_resources(service_id, resources)

    def __get_cms_last_tag_release(self):
        tag = self.__gencfg.get_last_stable_tag('unstable')
        assert tag is not None
        return 'tags/' + tag

    def __is_stopped_sandbox_task(self, task):
        return task.new_status in Status.Group.FINISH or task.new_status in Status.Group.BREAK

    def __get_active_services(self):
        return [service.get("_id", "") for service in
                self.__nanny_client.list_services_by_category(self.__nanny_category).get("result", [])]

    def __own_service(self, service):
        """
            Own service via putting self sandbox task id in nanny service label under key owner_sandbox_task.
        """
        info_attrs = self.__nanny_client.get_service_info_attrs(service)

        labels = [label for label in info_attrs['content']['labels'] if label['key'] != 'owner_sandbox_task']
        labels.append({
            'key': 'owner_sandbox_task',
            'value': self.__task_id
        })

        info_attrs['content']['labels'] = labels
        self.__save_info_attrs(service, info_attrs, comment='Set sandbox task {} as owner'.format(self.__task_id))

    def __get_alive_owners(self, services):
        """
           Get owners from nanny services labels with key owner_sandbox_task and cgeck if they are still alive
        """
        owners = []

        for service_id in services:
            info_attrs = self.__nanny_client.get_service_info_attrs(service_id)

            if info_attrs:
                labels = info_attrs['content']['labels']
                for label in labels:
                    if label['key'] == 'owner_sandbox_task':
                        sandbox_task_id = label['value']
                        task = channel.channel.sandbox.get_task(sandbox_task_id)
                        if task:
                            if not self.__is_stopped_sandbox_task(task):
                                logging.info('Task {task} is running, we have to wait it'.format(task=task))
                                owners.append(task.id)

                        break

        return owners

    def __activate_service(self, service_id):
        runtime_attrs = self.__nanny_client.get_service_runtime_attrs(service_id)
        snapshot_id = runtime_attrs["_id"]
        self.__nanny_client.set_snapshot_state(service_id,
                                               snapshot_id=snapshot_id,
                                               state="ACTIVE",
                                               comment='Starting testing instances',
                                               recipe="common")

    def __is_service_active(self, service_id):
        return self.__nanny_client.get_service_current_state(service_id) == "ONLINE"

    def __stop_services(self, services):
        logging.info('Stop services {services}'.format(services=services))

        for service_id in services:
            self.__nanny_client.shutdown(service_id, comment='Stop service')

    def __wait_services_for_status(self, services, status="ONLINE", timeout=300):
        waitfor = time.time() + timeout
        while time.time() < waitfor:
            active = True
            for service_id in services:
                active &= self.__nanny_client.get_service_current_state(service_id)\
                              .get("content", {})\
                              .get("summary", {})\
                              .get("value", "") == status
            if not active:
                time.sleep(30)
            else:
                return True
        raise SandboxTaskFailureError("Failed to set status {status} \
            to services {services} \
            in {timeout} seconds".format(services=services, status=status, timeout=timeout))

    def __copy_files(self, source_service, destination_service):
        source_resources = self.__nanny_client.get_service_resources(source_service)
        destination_resources = self.__nanny_client.get_service_resources(destination_service)

        for section in ['sandbox_files', 'static_files', 'url_files']:
            destination_resources['content'][section] = source_resources['content'][section]

        destination_resources['comment'] = 'Clone resources from service {}'.format(source_service)
        self.__nanny_client.update_service_resources(destination_service, destination_resources)

    def __patch_environment(self, service):
        resources = self.__nanny_client.get_service_resources(service)
        task_resources = sdk2.Resource.find(task_id=self.__build_base_task).limit(1000)
        for resource in task_resources:
            res = sdk2.Resource[resource.id]
            if res.type == images_basesearch_resources.IMGULTRA_SAAS_RTYSERVER_CONFIGS_BUNDLE.name:
                resource_data = sdk2.ResourceData(res)
                env_file = {'local_path': 'environment'}
                with open(os.path.join(str(resource_data.path), 'environment')) as f:
                    env_file['content'] = f.read().replace('CTYPE=prod', 'CTYPE=acceptance_test')
                if 'static_files' not in resources['content']:
                    resources['content']['static_files'] = list()
                resources['content']['static_files'].append(env_file)
                resources['comment'] = 'Patch environment static file'
                self.__nanny_client.update_service_resources(service, resources)
                return

    def __start_services(self, production_services, tesing_services):

        for production_service, tesing_service in zip(production_services, tesing_services):
            self.__own_service(tesing_service)
            self.__copy_files(production_service, tesing_service)
            self.__patch_environment(tesing_service)
            self.__set_media_recipes(tesing_service)
            self.__remove_tickets_integration_rules(tesing_service)
            if self.__build_base_task:
                self.__set_base_search_task(tesing_service, self.__build_base_task)
            if self.__build_models_task:
                self.__set_dynamic_models_task(tesing_service, self.__build_models_task)
            if self.__mr_index_config_task:
                self.__set_mrindex_config_task(tesing_service, self.__mr_index_config_task)
            self.__set_config_files(tesing_service, self.__config_files)
            self.__set_ssd_clear_cache_hook(tesing_service, self.__clear_cache_script)
            self.__activate_service(tesing_service)
        return tesing_services

    def __retry(self, func, tries=5, delay=2):
        while tries > -1:
            try:
                return func()
            except Exception as err:
                msg = '%s. retrying in %d seconds' % (str(err), delay)
                tries -= 1
                if tries == -1:
                    raise err
                logging.debug(msg)
                time.sleep(delay)

    def get_service_instance_urls(self, service_nanny_name):
        data = self.__retry(lambda: self.__nanny_client.get_service_current_instances(service_nanny_name))
        logging.info('%s: service_instances %s', service_nanny_name, json.dumps(data))
        urls = []
        for instance in data['result']:
            urls.append('http://{host}:{port}'.format(host=instance['container_hostname'], port=instance['port']))
        return urls

    def get_test_services(self):
        return self.__testing_services

    def set_test_services(self, services):
        self.__testing_services = services

    def find_running_tasks(self):
        alive_services = self.__get_active_services()
        logging.info('Alive services is {services}'.format(services=alive_services))
        tasks = self.__get_alive_owners(alive_services)
        return tasks

    def start(self):
        alive_services = self.__get_active_services()
        tasks = self.__get_alive_owners(alive_services)
        if tasks:
            raise SandboxTaskFailureError("Another acceptance tasks {} is running, \
                                           you must wait, but you don't".format(tasks))

        self.__stop_services(alive_services)
        self.__wait_services_for_status(self.__testing_services, "OFFLINE", self.__activation_timeout)
        self.__testing_services = self.__start_services(
            production_services=self.__production_services,
            tesing_services=self.__testing_services)
        self.__wait_services_for_status(self.__testing_services, "ONLINE", self.__activation_timeout)

    def stop(self):
        self.__stop_services(self.__testing_services)
        self.__wait_services_for_status(self.__testing_services, status="OFFLINE", timeout=self.__activation_timeout)
