# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.process import check_process_return_code
import os
import time

_INPUT_DATA_GROUP = "Input data params"


class ManualMrServer(SandboxStringParameter):
    name = 'server_override'
    description = 'Override mapreduce server from index location var'
    group = _INPUT_DATA_GROUP


class ManualMrPath(SandboxStringParameter):
    name = 'path_override'
    description = 'Override path prefix of base from index location var'
    group = _INPUT_DATA_GROUP


class IndexLocation(ResourceSelector):
    name = 'index_location'
    description = 'mapreduce server and path from IMAGES_STORE_INDEX_LOCATION resource'
    resource_type = resource_types.IMAGES_INDEX_LOCATION
    group = _INPUT_DATA_GROUP


_UPLOAD_ARGS_GROUP = "Upload args params"


class UploadSaasBin(ResourceSelector):
    name = 'saas_executable'
    description = 'saas executable'
    resource_type = resource_types.IMAGES_UPLOADSAAS_EXECUTABLE
    group = _UPLOAD_ARGS_GROUP


class SaasProxyServerName(SandboxStringParameter):
    name = 'saas_server'
    description = 'saas server[:saas port]'
    group = _UPLOAD_ARGS_GROUP


class Rps(SandboxIntegerParameter):
    name = 'rps'
    description = 'rps'
    default_value = 500
    group = _UPLOAD_ARGS_GROUP


class Deadline(SandboxIntegerParameter):
    name = 'deadline'
    description = 'Dead line for docs, in unix time. Default is now+3h.'
    group = _UPLOAD_ARGS_GROUP


class Inflow(SandboxIntegerParameter):
    name = 'inflow'
    description = 'inflow'
    default_value = 100
    group = _UPLOAD_ARGS_GROUP


class ExtraArgs(SandboxStringParameter):
    name = 'cmdline'
    description = 'additional command line parameters'
    group = _UPLOAD_ARGS_GROUP


class YtSpec(SandboxStringParameter):
    name = 'yt_spec'
    description = 'YT_SPEC'
    default_value = '{"resource_limits": { "user_slots": 1}}'
    group = _UPLOAD_ARGS_GROUP


class FinalMrServer(SandboxStringParameter):
    name = 'final_mr_server'
    description = 'account MapreduceServer and IndexLocationParameter'


class FinalMrPath(SandboxStringParameter):
    name = 'final_mr_path'
    description = 'account MapreduceBasePrefix and IndexLocationParameter'


class ImagesUploadSaas(SandboxTask):
    """
        Launch uploadSaas
    """
    type = 'IMAGES_UPLOAD_SAAS'
    input_parameters = (IndexLocation,
                        ManualMrServer,
                        ManualMrPath,
                        UploadSaasBin,
                        SaasProxyServerName,
                        Rps,
                        Deadline,
                        Inflow,
                        ExtraArgs,
                        YtSpec)

    # export names in stx for easy creating subtask
    INDEX_LOCATION = IndexLocation.name
    SAAS_BASESEARCH_SERVER = SaasProxyServerName.name
    UPLOAD_SAAS_BIN = UploadSaasBin.name

    @staticmethod
    def __get_saas_host_and_port(url):
        assert isinstance(url, (str, unicode))
        url = url.replace("http://", "")
        parts = url.split(":")
        if len(parts) == 1:
            return parts[0], 80
        if len(parts) == 2:
            return parts[0], parts[1]
        raise Exception("Can't parse saas url {}".format(url))

    def patch_parameters(self):
        server = None
        path = None

        if utils.get_or_default(self.ctx, IndexLocation):
            location_file = utils.sync_resource(utils.get_or_default(self.ctx, IndexLocation))
            (server, path) = fu.read_file(location_file).split()

        if utils.get_or_default(self.ctx, ManualMrServer):
            server = utils.get_or_default(self.ctx, ManualMrServer)

        if utils.get_or_default(self.ctx, ManualMrPath):
            path = utils.get_or_default(self.ctx, ManualMrPath)

        self.ctx[FinalMrServer.name] = server
        self.ctx[FinalMrPath.name] = path

        if not utils.get_or_default(self.ctx, Deadline):
            self.ctx[Deadline.name] = int(time.time()) + 3 * 60 * 60

    def on_prepare(self):
        # do it before running to fill context and have a good footer
        self.patch_parameters()

    def on_execute(self):
        api_key = self.get_vault_data("IMAGES-ROBOT", "saas_api_key")
        yt_token = self.get_vault_data("IMAGES-ROBOT", "yt_token")

        upload_saas_bin = utils.sync_resource(utils.get_or_default(self.ctx, UploadSaasBin))
        saas_host, saas_port = self.__get_saas_host_and_port(utils.get_or_default(self.ctx, SaasProxyServerName))
        mr_server = utils.get_or_default(self.ctx, FinalMrServer)
        mr_path = utils.get_or_default(self.ctx, FinalMrPath)
        rps = utils.get_or_default(self.ctx, Rps)
        dead_line = utils.get_or_default(self.ctx, Deadline)
        in_flow = utils.get_or_default(self.ctx, Inflow)
        yt_spec = utils.get_or_default(self.ctx, YtSpec)
        extra_args = utils.get_or_default(self.ctx, ExtraArgs)

        cmdline = [upload_saas_bin,
                   '--saas-server', '{}'.format(saas_host),
                   '--saas-port', '{}'.format(saas_port),
                   '--server', '{}'.format(mr_server),
                   '--portion', '{}/rt_robot/portion'.format(mr_path),
                   '--errors', '{}/sandbox/rt_robot/upload.err'.format(mr_path, self.id),
                   '--stats', '{}/sandbox/rt_robot/upload.stats'.format(mr_path, self.id),
                   '--rps', '{}'.format(rps),
                   '--is-realtime',
                   '--from-rt-ultra',
                   '--dead-line', '{}'.format(dead_line),
                   '--api-key', api_key,
                   '--inflow', '{}'.format(in_flow)]
        cmdline += '{}'.format(extra_args).split()

        env = os.environ.copy()
        env['YT_SPEC'] = yt_spec
        env['MR_RUNTIME'] = 'YT'
        env['MR_OPT'] = 'stderrlevel=1'
        env['YT_TOKEN'] = yt_token

        process = run_process(cmdline, log_prefix='uploadSaas', environment=env)
        check_process_return_code(process)

    @property
    def footer(self):
        if FinalMrServer.name in self.ctx and FinalMrPath.name in self.ctx:
            return (''' <h4>
                            <a href="https://yt.yandex-team.ru/{server}/#page=navigation&path={path}">
                                Index in YT
                            </a>
                        </h4>
                            <a href="https://yt.yandex-team.ru/{server}/#page=operation&filter=TUploadOp&state=all">
                                Last operations
                            </a>
                        <br>
                    '''.format(server=self.ctx[FinalMrServer.name],
                               path=self.ctx[FinalMrPath.name]))


__Task__ = ImagesUploadSaas
