import json
import logging
import os

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths

from sandbox.projects import resource_types
from sandbox.projects.common.mediasearch import ban as media_ban
from sandbox.projects.images.bans import yt_task
from sandbox.projects.images.bans import resources as images_bans_resources


class ImagesReleaseAntipornoBan(yt_task.YtTask, media_ban.ImagesBaseReleaseBanTask):
    """
        Builds antiporno ban resources for Yandex.Images service
    """

    type = "IMAGES_RELEASE_ANTIPORNO_BAN"

    input_parameters = \
        media_ban.ImagesBaseReleaseBanTask.create_input_parameters(enable_semaphore=True) + \
        yt_task.YtTask.create_params("hahn.yt.yandex.net", "//home/antiporno/metasearch_fixlists/merged/img")

    vault_owner = "IMAGES-BAN"

    ban_resources = (
        images_bans_resources.IMAGES_MIDDLESEARCH_ANTIPORNO_PORNO_BAN_FILE,
        images_bans_resources.IMAGES_MIDDLESEARCH_ANTIPORNO_GRUESOME_BAN_FILE,
        images_bans_resources.IMAGES_MIDDLESEARCH_ANTIPORNO_PERVERSION_BAN_FILE,
    )
    version_resource = images_bans_resources.IMAGES_MIDDLESEARCH_ANTIPORNO_BAN_VERSION

    release_subject = "images/middle/antiporno-data-{timestamp}"
    release_comment = "daily images ban file (antiporno)"
    release_resources = ban_resources + (version_resource,)
    push_signal_name = "antiporno_ban"

    def _build_ban(self):
        # check
        version_path = self.abs_path("version")
        with open(version_path, "w") as version_file:
            version_file.write(self._yt_check())

        if self._skip_build(version_path, self.version_resource):
            logging.info("Nothing to release")
            return 0

        self._register_ban(self.descr, version_path, self.version_resource)

        # build
        urls_dir = self.abs_path("urls")
        hashes_dir = self.__get_hashes_dir()

        paths.make_folder(urls_dir, True)
        paths.make_folder(hashes_dir, True)

        output_files = {}
        try:
            for ban_resource in self.ban_resources:
                ban_reason = ban_resource.ban_reason
                urls_path = self.__antiporno_path(urls_dir, ban_reason)
                output_files[ban_reason] = open(urls_path, "w")

            for key, value in self._yt_get(["key", "value"]):
                value = json.loads(value)
                for ban_reason, ban_file in output_files.iteritems():
                    if value.get("Is" + ban_reason.title()):
                        ban_file.write(key + "\n")
        finally:
            for output_file in output_files.itervalues():
                output_file.close()

        hasher_tool = self._tool(images_bans_resources.IMAGES_URL2FASTBAN_EXECUTABLE)
        total_size = 0
        for ban_resource in self.ban_resources:
            ban_reason = ban_resource.ban_reason
            urls_path = self.__antiporno_path(urls_dir, ban_reason)
            hashes_path = self.__get_hashes_resource_path(ban_resource)

            logging.info("Generating {}...".format(hashes_path))

            process.run_process([
                hasher_tool,
                "--mode", "hashes",
                "--url-type", "url",
                "--ban-reason", ban_reason,
                "--ban-type", "doc",
                "--input", urls_path,
                "--output", hashes_path,
            ], outputs_to_one_file=False, log_prefix="hasher.{}".format(ban_reason))

            self._register_ban(self.descr, hashes_path, ban_resource)
            self._register_ban(self.descr, urls_path, resource_types.OTHER_RESOURCE)

            total_size += os.stat(hashes_path).st_size

        self._set_release_signal()

        return total_size

    def _test_ban(self, build_task_id):
        return [self._test_task(build_task_id, ban_resource) for ban_resource in self.ban_resources]

    def __antiporno_path(self, urls_dir, ban_reason):
        return os.path.join(urls_dir, "urls.{}".format(ban_reason))

    def __get_hashes_dir(self):
        return self.abs_path("hashes")

    def __get_hashes_resource_path(self, ban_resource):
        return os.path.join(self.__get_hashes_dir(), ban_resource.basename)


__Task__ = ImagesReleaseAntipornoBan
