import os
import logging

from sandbox.sandboxsdk import process

from sandbox.projects import resource_types
from sandbox.projects.common.mediasearch import ban as media_ban
from sandbox.projects.images.bans import yt_task
from sandbox.projects.images.bans import resources as images_bans_resources

from ban_diff import BanDiff, BanDiffParams


class ImagesReleaseAntispamBan(yt_task.YtTask, media_ban.ImagesBaseReleaseBanTask):
    """
        Builds antispam ban resources for Yandex.Images service
    """

    type = "IMAGES_RELEASE_ANTISPAM_BAN"
    client_tags = media_ban.ImagesBaseReleaseBanTask.client_tags & yt_task.YtTask.client_tags

    input_parameters = \
        media_ban.ImagesBaseReleaseBanTask.create_input_parameters(enable_semaphore=True) + \
        yt_task.YtTask.create_params("banach.yt.yandex.net", "//home/imgdev/images/antispam/archive/ban_diff") + \
        BanDiffParams.create_input()

    vault_owner = "IMAGES-BAN"

    ban_resource = images_bans_resources.IMAGES_MIDDLESEARCH_ANTISPAM_OWNERS_BAN_FILE
    version_resource = images_bans_resources.IMAGES_MIDDLESEARCH_ANTISPAM_BAN_VERSION

    release_subject = "images/middle/antispam-data-{timestamp}"
    release_comment = "daily images ban file (antispam)"
    release_resources = (ban_resource, version_resource)
    push_signal_name = "antispam_ban"

    def _build_ban(self):
        import yt.wrapper.client
        ban_diff_params = BanDiffParams()
        ban_diff_params.get_from_ctx(self.ctx, self._yt_table())
        config = {"token": self.get_vault_data(self.vault_owner, "yt_token")}
        client = yt.wrapper.client.Yt(proxy=self._yt_proxy(), config=config)
        ban_diff_maker = BanDiff(client, ban_diff_params)

        # check
        version_path = self.abs_path("antispam.version")
        ban_version = self._yt_check() if ban_diff_params.disable else ban_diff_maker.make_version()
        with open(version_path, "w") as version_file:
            version_file.write(ban_version)

        if self._skip_build(version_path, self.version_resource):
            logging.info("Nothing to release")
            return 0

        self._register_ban(self.descr, version_path, self.version_resource)

        # build
        ban_diff_maker.make_diff()
        ban_url_count = 0
        urls_path = self.abs_path("antispam.urls")
        with open(urls_path, "w") as urls_file:
            for key in self._yt_get(["key"]):
                urls_file.write(key + "\n")
                ban_url_count += 1
        logging.info("Read {} owners from ban diff".format(ban_url_count))

        hasher_tool = self._tool(images_bans_resources.IMAGES_URL2FASTBAN_EXECUTABLE)

        hashes_path = self.abs_path(self.ban_resource.basename)
        process.run_process([
            hasher_tool,
            "--mode", "hashes",
            "--url-type", "owner",
            "--input", urls_path,
            "--output", hashes_path,
        ], log_prefix="hasher")

        self._register_ban(self.descr, urls_path, resource_types.OTHER_RESOURCE)
        self._register_ban(self.descr, hashes_path, self.ban_resource)
        self._set_release_signal()

        return os.stat(hashes_path).st_size

    def _test_ban(self, build_task_id):
        return [self._test_task(build_task_id, self.ban_resource)]


__Task__ = ImagesReleaseAntispamBan
