import logging
import os

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process

from sandbox.projects.common.mediasearch import ban as media_ban
from sandbox.projects.images.bans import resources as images_bans_resources


class MrServerParameter(parameters.SandboxStringParameter):
    name = 'mr_server'
    description = 'MapReduce server'
    default_value = 'titan00.search.yandex.net:8013'
    required = True


class MrUserParameter(parameters.SandboxStringParameter):
    name = 'mr_user'
    description = 'MapReduce user'
    default_value = 'images-money'
    required = True


class BanMrTableParameter(parameters.SandboxStringParameter):
    name = 'ban_table'
    description = 'Ban table'
    default_value = 'images/money/query_ban'
    required = True


class ImagesReleaseCommercialQueryBan(media_ban.ImagesBaseReleaseBanTask):
    '''
        Builds commercial query ban trie resource for Yandex.Images service.
    '''

    type = 'IMAGES_RELEASE_COMMERCIAL_QUERY_BAN'

    input_parameters = \
        media_ban.ImagesBaseReleaseBanTask.create_input_parameters(enable_semaphore=True) + ( \
        MrServerParameter,
        MrUserParameter,
        BanMrTableParameter
    )

    ban_resource = images_bans_resources.IMAGES_MIDDLESEARCH_COMMERCIAL_QUERY_BAN_TRIE
    version_resource = images_bans_resources.IMAGES_MIDDLESEARCH_COMMERCIAL_QUERY_BAN_VERSION

    release_subject = 'images/middle/commercial-query-ban-{timestamp}'
    release_comment = 'images commercial query ban release'
    release_mailto = 'images-data-releases@yandex-team.ru'
    release_resources = (ban_resource, version_resource)

    def _build_ban(self):
        version_path = self.abs_path('version')
        with open(version_path, 'w') as version_file:
            self.__process(
                'GetCommercialQueryBanState',
                stdout=version_file
            )

        if self._skip_build(version_path, self.version_resource):
            logging.info('Nothing to release')
            return

        self._register_ban(self.descr, version_path, self.version_resource)

        trie_path = self.abs_path(images_bans_resources.IMAGES_MIDDLESEARCH_COMMERCIAL_QUERY_BAN_TRIE.basename)
        self.__process(
            'WriteCommercialQueryBanTrie',
            args=[
                '--user', self.ctx[MrUserParameter.name],
                '--trie-file', trie_path
            ]
        )
        self._register_ban(self.descr, trie_path, self.ban_resource)
        self._set_release_signal()

        return os.stat(trie_path).st_size

    def _test_ban(self, build_task_id):
        return [self._test_task(build_task_id, self.ban_resource, rule_name="ImgCommercial")]

    def __process(self, mode, args=[], stdout=None):
        trie_builder_tool = self._tool(images_bans_resources.IMAGES_COMMERCIAL_BAN_BUILDER_EXECUTABLE)
        process.run_process([
            trie_builder_tool,
            mode,
            '--server', self.ctx[MrServerParameter.name],
            '--ban-table', self.ctx[BanMrTableParameter.name]
        ] + args, stdout=stdout, outputs_to_one_file=False, log_prefix='commercial_ban.{}'.format(mode))


__Task__ = ImagesReleaseCommercialQueryBan
