import os
import logging

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import svn

from sandbox.projects.common import utils
from sandbox.projects.common.mediasearch import ban as media_ban
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.images.bans import resources as images_bans_resources


_BAN_RESOURCE = {
    search_settings.INDEX_MAIN: images_bans_resources.IMAGES_MIDDLESEARCH_QUERY_URL_BAN_FILE,
    search_settings.INDEX_QUICK: images_bans_resources.IMAGES_MIDDLESEARCH_QUERY_URL_QUICK_BAN_FILE,
}

_VERSION_RESOURCE = {
    search_settings.INDEX_MAIN: images_bans_resources.IMAGES_MIDDLESEARCH_QUERY_BAN_VERSION,
    search_settings.INDEX_QUICK: images_bans_resources.IMAGES_MIDDLESEARCH_QUERY_QUICK_BAN_VERSION,
}


class IndexTypeParameter(parameters.SandboxStringParameter):
    name = 'index_type'
    description = 'Index type'
    choices = [
        ('Main', search_settings.INDEX_MAIN),
        ('Quick', search_settings.INDEX_QUICK),
    ]
    default_value = search_settings.INDEX_MAIN


class MrServerParameter(parameters.SandboxStringParameter):
    name = "mr_server"
    description = "Mapreduce server"
    default_value = "banach.yt.yandex.net"
    required = True


class MrTableParameter(parameters.SandboxStringParameter):
    name = "mr_table"
    description = "Mapreduce data table"
    default_value = "//home/imgdev/images/config/queryurlbandata"
    required = True


class IndexStateParameter(parameters.SandboxStringParameter):
    name = "index_state"
    description = "Index State"
    default_value = "production"
    required = True


class MrTolokaParameter(parameters.ResourceSelector):
    name = 'mr_toloka_executable'
    description = 'MrToloka executable'
    resource_type = images_bans_resources.IMAGES_MRTOLOKA_EXECUTABLE


class LanguageDataParameter(parameters.SandboxArcadiaUrlParameter):
    name = 'language_data_url'
    description = 'SVN url of folder with language data'
    default_value = 'arcadia:/arc/trunk/arcadia_tests_data/wizard/language'
    required = True


class SynnormDataParameter(parameters.SandboxArcadiaUrlParameter):
    name = 'synnorm_data_url'
    description = 'SVN url of folder with language data'
    default_value = 'arcadia:/arc/trunk/arcadia_tests_data/wizard/synnorm'
    required = True


class ImagesReleaseQueryUrlBan(media_ban.ImagesBaseReleaseBanTask):
    """
        Builds query ban resources for Yandex.Images service
    """

    type = "IMAGES_RELEASE_QUERYURL_BAN"

    input_parameters = \
        media_ban.ImagesBaseReleaseBanTask.create_input_parameters(enable_semaphore=True) + ( \
        IndexTypeParameter,
        MrServerParameter,
        IndexStateParameter,
        MrTableParameter,
        MrTolokaParameter,
        LanguageDataParameter,
        SynnormDataParameter,
    )

    @property
    def release_resources(self):
        index_type = utils.get_or_default(self.ctx, IndexTypeParameter)
        return (_BAN_RESOURCE[index_type], _VERSION_RESOURCE[index_type])

    @property
    def release_subject(self):
        index_type = utils.get_or_default(self.ctx, IndexTypeParameter)
        return "images/middle/queryurl-{}-data".format(index_type) + "-{timestamp}"

    @property
    def release_comment(self):
        index_type = utils.get_or_default(self.ctx, IndexTypeParameter)
        return "daily images ban file ({}, queryurl)".format(index_type)

    @property
    def push_signal_name(self):
        index_type = utils.get_or_default(self.ctx, IndexTypeParameter)
        return "release_{}_queryurl_ban".format(index_type)

    def _build_ban(self):
        index_type = utils.get_or_default(self.ctx, IndexTypeParameter)

        # check
        version_resource = _VERSION_RESOURCE[index_type]
        version_path = self.abs_path("version")
        with open(version_path, "w") as version_file:
            self.__mr_toloka("CheckQueryBanMRTable", stdout=version_file)

        if self._skip_build(version_path, version_resource):
            logging.info("Nothing to release")
            return 0

        self._register_ban(self.descr, version_path, version_resource)

        # build
        geobase3_resource = utils.get_and_check_last_resource_with_attribute(images_bans_resources.IMAGES_GEOBASE5_SNAPSHOT)
        geobase3_path = self.sync_resource(geobase3_resource.id)

        language_path = self.abs_path("language")
        svn.Arcadia.export(self.ctx[LanguageDataParameter.name], language_path)

        synnorm_path = self.abs_path("synnorm")
        svn.Arcadia.export(self.ctx[SynnormDataParameter.name], synnorm_path)

        ban_resource = _BAN_RESOURCE[index_type]
        ban_path = self.abs_path(ban_resource.basename)

        self.__mr_toloka("CreateQueryBanTrie", [
            "--wizard-language-dir", language_path,
            "--wizard-synnorm-dir", synnorm_path,
            "--geodata3", geobase3_path,
            "--thread-count", "4",
            "--job-count-multiplier", "2",
            "--output-file", ban_path
        ])
        self._register_ban(self.descr, ban_path, ban_resource)
        self._set_release_signal()

        return os.stat(ban_path).st_size

    def _test_ban(self, build_task_id):
        ban_resource = _BAN_RESOURCE[utils.get_or_default(self.ctx, IndexTypeParameter)]
        return [self._test_task(build_task_id, ban_resource, rule_name="ImgQueryUrlBan", max_removal_delta=0.5)]

    def __mr_toloka(self, cmd, args=[], stdout=None):
        mrtoloka_tool = self._tool(images_bans_resources.IMAGES_MRTOLOKA_EXECUTABLE, MrTolokaParameter.name)
        yt_token = self.get_vault_data("IMAGES-BAN", "yt_token")
        process.run_process([
            mrtoloka_tool,
            cmd,
            "--server", self.ctx[MrServerParameter.name],
            "--index-prefix", _BAN_RESOURCE[utils.get_or_default(self.ctx, IndexTypeParameter)].index_prefix,
            "--queryurlbandata-table", utils.get_or_default(self.ctx, MrTableParameter),
            "--index-state", self.ctx[IndexStateParameter.name]
            ] + args,
            environment={"MR_RUNTIME": "YT", "YT_TOKEN": yt_token},
            stdout=stdout, outputs_to_one_file=False, log_prefix="mr_toloka.{}".format(cmd))


__Task__ = ImagesReleaseQueryUrlBan
