import logging
import os

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.common.mediasearch import ban as media_ban
from sandbox.projects.images.bans import yt_task
from sandbox.projects.images.bans import resources as images_bans_resources
from sandbox.projects.common import utils


class RimdbPrefixParameter(parameters.SandboxStringParameter):
    name = "rimdb_prefix"
    description = "RIMDB prefix"
    default_value = "//home/images"
    required = True


class RimdbStateParameter(parameters.SandboxStringParameter):
    name = "rimdb_state"
    description = "RIMDB state"
    required = True


class OperationYtPoolParameter(parameters.SandboxStringParameter):
    name = "operation_pool"
    description = "Build ban trie operation yt pool"
    default_value = "images_bases"
    required = True


class ImagesReleaseRimRelatedDocumentsBan(yt_task.YtTask, media_ban.ImagesBaseReleaseBanTask):
    """
        Builds ban resource for related documents of RIM documents
    """

    type = "IMAGES_RELEASE_RIM_RELATED_DOCUMENTS_BAN"
    client_tags = media_ban.ImagesBaseReleaseBanTask.client_tags & yt_task.YtTask.client_tags

    vault_owner = "IMAGES-BAN"

    input_parameters = \
        media_ban.ImagesBaseReleaseBanTask.create_input_parameters(enable_semaphore=True) + \
        yt_task.YtTask.create_params("banach.yt.yandex.net", "//home/images/relatedimages/ban_related_docs.list") + \
        (RimdbPrefixParameter, RimdbStateParameter, OperationYtPoolParameter)

    release_subject = "images/middle/rim-related-documents-{timestamp}"
    release_comment = "daily images RIM ban file (related documents)"
    push_signal_name = "rim_related_documents_ban"

    ban_resource = images_bans_resources.IMAGES_RIM_BAN_RELATEDDOCS_TRIE
    version_resource = images_bans_resources.IMAGES_RIM_BAN_RELATEDDOCS_VERSION
    release_resources = (ban_resource, version_resource)

    def _build_ban(self):
        # check
        rimdb_state = utils.get_or_default(self.ctx, RimdbStateParameter)

        version_path = self.abs_path(self.version_resource)

        with open(version_path, "w") as version_file:
            version_file.write(self._yt_check() + "|" + rimdb_state)

        if self._skip_build(version_path, self.version_resource):
            logging.info("Nothing to release")
            return 0

        yt_token = self.get_vault_data(self.vault_owner, "yt_token")

        trie_path = self.abs_path(self.ban_resource.basename)
        rimdb_tools = self._tool(resource_types.IMAGES_RIMDB_TOOLS)
        banned_path = self.abs_path("rim.relateddocs.banned")
        notbanned_path = self.abs_path("rim.relateddocs.notbanned")

        process.run_process([
            rimdb_tools,
            "RelationsBanTrie",
            "--server", utils.get_or_default(self.ctx, yt_task.YtProxyParameter),
            "--rimdb-prefix", utils.get_or_default(self.ctx, RimdbPrefixParameter),
            "--rimdb-state", utils.get_or_default(self.ctx, RimdbStateParameter),
            "--ban", utils.get_or_default(self.ctx, yt_task.YtTableParameter),
            "--banned", banned_path,
            "--notbanned", notbanned_path,
            "--trie", trie_path],
            environment={
                "YT_TOKEN": yt_token,
                "YT_LOG_LEVEL": "INFO",
                "YT_POOL": utils.get_or_default(self.ctx, OperationYtPoolParameter)
            },
            outputs_to_one_file=False,
            log_prefix="rim")

        self._register_ban(self.descr, version_path, self.version_resource)
        self._register_ban(self.descr, banned_path, resource_types.OTHER_RESOURCE)
        self._register_ban(self.descr, notbanned_path, resource_types.OTHER_RESOURCE)
        self._register_ban(self.descr, trie_path, self.ban_resource)

        return os.stat(trie_path).st_size

    def _test_ban(self, build_task_id):
        return []


__Task__ = ImagesReleaseRimRelatedDocumentsBan
