import os
import time

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.common.mediasearch import ban as media_ban
from sandbox.projects.images.bans import resources as images_bans_resources


class MrServerParameter(parameters.SandboxStringParameter):
    name = "mr_server"
    description = "Mapreduce server"
    required = True
    default_value = "banach"


class MrPrefixParameter(parameters.SandboxStringParameter):
    name = "mr_prefix"
    description = "Mapreduce table prefix"
    required = True
    default_value = "images"


class MrPoolParameter(parameters.SandboxStringParameter):
    name = "mr_pool"
    description = "Mapreduce pool"
    required = False
    default_value = "images_bases"


class ImagesReleaseRottenHostBan(media_ban.ImagesBaseReleaseBanTask):
    """
        Builds rotten ban resources for Yandex.Images service
    """

    type = "IMAGES_RELEASE_ROTTEN_HOST_BAN"

    input_parameters = \
        media_ban.ImagesBaseReleaseBanTask.create_input_parameters(enable_semaphore=True) + ( \
        MrServerParameter,
        MrPrefixParameter,
        MrPoolParameter,
    )

    release_subject = "images/middle/rottenhost-data-{timestamp}"
    release_comment = "daily images ban file (rotten hosts)"
    ban_resource = images_bans_resources.IMAGES_MIDDLESEARCH_ROTTEN_HOSTS_BAN_FILE
    version_resource = images_bans_resources.IMAGES_MIDDLESEARCH_ROTTEN_HOSTS_BAN_VERSION
    release_resources = (ban_resource, version_resource)
    push_signal_name = "rotten_host_ban"

    def _build_ban(self):
        # check
        # Note: rottenhosts utility always creates a new ban content
        version_path = self.abs_path("rotten-hosts.version")
        with open(version_path, "w") as version_file:
            timestamp = int(time.time())
            version_file.write("{}".format(timestamp))
        self._register_ban(self.descr, version_path, self.version_resource)

        # build
        rottenhosts_tool = self._tool(images_bans_resources.IMAGES_ROTTENHOSTS_EXECUTABLE)
        hosts_path = self.abs_path("rotten.hosts")
        with open(hosts_path, "w") as hosts_file:
            environment = {
                "MR_RUNTIME": "YT",
                "YT_PREFIX": "//home/",
                "MR_TMP": "//tmp",
                "YT_USE_YAMR_STYLE_PREFIX": "1",
                "YT_USE_YAMR_DEFAULTS": "1",
                "YT_TOKEN": self.get_vault_data("IMAGES-BAN", "yt_token"),
            }
            mr_pool = self.ctx[MrPoolParameter.name]
            if mr_pool:
                environment["YT_POOL"] = mr_pool
            process.run_process([
                 rottenhosts_tool,
                 "--server", self.ctx[MrServerParameter.name],
                 "--prefix", self.ctx[MrPrefixParameter.name],
            ],
            environment=environment, stdout=hosts_file, outputs_to_one_file=False, log_prefix="rottenhosts")

        hasher_tool = self._tool(images_bans_resources.IMAGES_URL2FASTBAN_EXECUTABLE)
        hashes_path = self.abs_path(self.ban_resource.basename)
        process.run_process([
            hasher_tool,
            "--mode", "hashes",
            "--url-type", "host,image_host",
            "--ban-reason", "common",
            "--ban-type", "url",
            "--input", hosts_path,
            "--output", hashes_path,
        ], log_prefix="hasher")

        self._register_ban(self.descr, hosts_path, resource_types.OTHER_RESOURCE)
        self._register_ban(self.descr, hashes_path, self.ban_resource)
        self._set_release_signal()

        return os.stat(hashes_path).st_size

    def _test_ban(self, build_task_id):
        return [self._test_task(build_task_id, self.ban_resource)]


__Task__ = ImagesReleaseRottenHostBan
