import logging

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import utils
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images import responses
from sandbox.projects.images.bans import resources as images_bans_resources
from sandbox.projects.images.metasearch import ImagesGetMiddlesearchResponses as responses_task
from sandbox.projects.images.metasearch import task as metasearch_task
from sandbox.projects.images.resources import task as resources_task


_TEST_TASKS = 'test_tasks'


class FirstTrieParameter(parameters.ResourceSelector):
    name = 'ban1_resource_id'
    description = 'First trie'
    resource_type = images_bans_resources.IMAGES_MIDDLESEARCH_COMMERCIAL_TRIE
    required = True


class SecondTrieParameter(parameters.ResourceSelector):
    name = 'ban2_resource_id'
    description = 'Second trie'
    resource_type = images_bans_resources.IMAGES_MIDDLESEARCH_COMMERCIAL_TRIE
    required = True


class MaxDeltaParameter(parameters.SandboxFloatParameter):
    name = 'max_removal_delta'
    description = 'Maximum allowed removal delta of commercial data (in percents)'
    default_value = 0.05


class ImagesTestCommercialTrie(resources_task.ImagesProductionResourcesTask, task.SandboxTask):
    """
        Compare old and new commercial tries

        Verify number of commercial documents found
    """

    type = "IMAGES_TEST_COMMERCIAL_TRIE"

    input_parameters = (
        FirstTrieParameter,
        SecondTrieParameter,
        MaxDeltaParameter,
    )

    def on_execute(self):
        if _TEST_TASKS not in self.ctx:
            self.ctx[_TEST_TASKS] = [
                self._test_task(self.ctx[FirstTrieParameter.name], "first"),
                self._test_task(self.ctx[SecondTrieParameter.name], "second"),
            ]
        utils.check_subtasks_fails(fail_on_first_failure=True)

        stats1, stats2 = [self.__get_stats(task_id) for task_id in self.ctx[_TEST_TASKS]]
        self.__verify_commercial_data(stats1, stats2)

    def __get_stats(self, task_id):
        task = channel.sandbox.get_task(task_id)
        return task.ctx.get("stats") or task.ctx.get(metasearch_task.BaseMiddlesearchTask.evlog_stats_key)

    def __verify_commercial_data(self, stats1, stats2):
        commercial_data1 = stats1["commercial_data"]
        commercial_data2 = stats2["commercial_data"]
        logging.info("commercial_data: {} -> {}".format(commercial_data1, commercial_data2))
        if not commercial_data1:
            raise errors.SandboxTaskFailureError("Strange delta 0 -> {}".format(commercial_data2))
        commercial_delta = abs(float(commercial_data2 - commercial_data1) / commercial_data1)
        if commercial_delta > self.ctx[MaxDeltaParameter.name]:
            raise errors.SandboxTaskFailureError("Delta limit exceeded {}%".format(commercial_delta * 100))

    def _test_task(self, ban_resource_id, description):
        index_type = media_settings.INDEX_MAIN
        sub_ctx = {
            metasearch_task.MIDDLESEARCH_PARAMS.Binary.name: self._get_middlesearch_executable(),
            metasearch_task.MIDDLESEARCH_PARAMS.Config.name: self._get_middlesearch_config(),
            metasearch_task.MIDDLESEARCH_PARAMS.Data.name: self._get_middlesearch_data(),

            metasearch_task.BASESEARCH_PARAMS.Binary.name: self._get_basesearch_executable(index_type),
            metasearch_task.BASESEARCH_PARAMS.Config.name: self._get_basesearch_config(index_type),

            metasearch_task.SNIPPETIZER_PARAMS.Binary.name: self._get_basesearch_executable(index_type),
            metasearch_task.SNIPPETIZER_PARAMS.Config.name: self._get_basesearch_config(index_type),

            responses.RESPONSE_SAVER_PARAMS.QueriesParameter.name: self._get_middlesearch_queries(media_settings.INDEX_MIDDLE, index_type),
            responses.RESPONSE_SAVER_PARAMS.QueriesLimitParameter.name: 10000,

            responses_task.CustomBanParameter.name: ban_resource_id,
            responses_task.EnableMarketParameter.name: True,
        }
        sub_task = self.create_subtask(
            task_type=responses_task.ImagesGetMiddlesearchResponses.type,
            description="{}, {}".format(self.descr, description),
            input_parameters=sub_ctx,
            arch=sandboxapi.ARCH_LINUX,
            model="e5-2650",
            inherit_notifications=True
        )
        return sub_task.id


__Task__ = ImagesTestCommercialTrie
