import logging

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import dolbilka
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common import utils
from sandbox.projects.images.metasearch import task as metasearch_task
from sandbox.projects.images.metasearch import ImagesTestMiddlesearchPerformance as performance_task
from sandbox.projects.images.resources import task as resources_task


class FirstBanParameter(parameters.ResourceSelector):
    name = 'ban1_resource_id'
    description = 'First ban'
    resource_type = media_settings.ImagesSettings.middlesearch_data_resources(media_settings.INDEX_MIDDLE)
    required = True


class SecondBanParameter(parameters.ResourceSelector):
    name = 'ban2_resource_id'
    description = 'Second ban'
    resource_type = media_settings.ImagesSettings.middlesearch_data_resources(media_settings.INDEX_MIDDLE)
    required = True


class RuleNameParameter(parameters.SandboxStringParameter):
    name = 'rule_name'
    description = 'Rearrange rule'
    required = True


class MaxErrorsCountParameter(parameters.SandboxIntegerParameter):
    name = 'max_errors_count'
    description = 'Maximum allowed number of errors'
    default_value = 200


class MaxRemovalDeltaParameter(parameters.SandboxFloatParameter):
    name = 'max_removal_delta'
    description = 'Maximum allowed removal delta (in percents)'
    default_value = 0.4


class TestMultiplierParameter(parameters.SandboxIntegerParameter):
    name = 'test_multiplier'
    description = 'number of simultaneously run copies of each ban test'
    default_value = 3


class MaxRepeatsCountParameter(parameters.SandboxIntegerParameter):
    name = 'max_repeats_count'
    description = 'Maximum allowed number of sequence repetitions of each test iteration'
    default_value = 10


class SubTaskRamParameter(parameters.SandboxStringParameter):
    name = 'subtask_ram'
    description = 'list of required ram in Gb of each test copy (like 200,100,100)'
    default_value = '40,40,40'


class ImagesTestMiddlesearchBan(resources_task.ImagesProductionResourcesTask, task.SandboxTask):
    """
        Compare old and new ban files
    """

    type = "IMAGES_TEST_MIDDLESEARCH_BAN"

    cores = 1
    required_ram = 4096
    execution_space = 4096

    input_parameters = (
        FirstBanParameter,
        SecondBanParameter,
        RuleNameParameter,
        MaxErrorsCountParameter,
        MaxRemovalDeltaParameter,
        TestMultiplierParameter,
        MaxRepeatsCountParameter
    )

    def on_execute(self):
        bans = {"first": FirstBanParameter.name, "second": SecondBanParameter.name}
        stats = {}
        verified = {}
        removals = {}
        rule = self.ctx[RuleNameParameter.name]
        mult_len = self.ctx.get(TestMultiplierParameter.name, TestMultiplierParameter.default_value)
        mult_rams = self.ctx.get(SubTaskRamParameter.name, SubTaskRamParameter.default_value).replace(",", " ").split() + [""] * mult_len
        mult_params = [(mult_id, mult_rams[mult_id]) for mult_id in xrange(mult_len)]
        for test_id in xrange(self.ctx.get(MaxRepeatsCountParameter.name, MaxRepeatsCountParameter.default_value)):
            new_tasks = {side: {self._get_task_key(side, test_id, mult_id): mult_ram for mult_id, mult_ram in mult_params}
                         for side in bans if not verified.get(side)}
            if not new_tasks:
                break

            now_running = []
            successful = {}
            for side in new_tasks:
                for task_key in new_tasks[side]:
                    if task_key not in self.ctx:
                        self.ctx[task_key] = self._test_task(self.ctx[bans[side]], task_key, new_tasks[side][task_key])
                    test_task_id = self.ctx[task_key]
                    test_task = channel.sandbox.get_task(test_task_id)
                    if not utils.is_task_stop_executing(test_task):
                        now_running.append(test_task)
                    elif test_task.new_status == self.Status.SUCCESS:
                        successful.setdefault(side, set()).add(task_key)
                        stats[task_key] = self._get_stats(self.ctx[task_key])
                        if self._verify_errors(stats[task_key], side):
                            verified.setdefault(side, set()).add(task_key)
                            removals[task_key] = stats[task_key]["removals"][rule]

            if self._check_removal(verified, removals, rule):
                for test_task in now_running:
                    try:
                        task_id = test_task.id
                        logging.info('Stopping excess task: {}'.format(task_id))
                        channel.sandbox.server.cancel_task(task_id)
                    except:
                        pass
                return

            if now_running:
                self.wait_any_task_completed(now_running, None, True)
                return

            if min([len(successful.get(side, set())) for side in new_tasks]) == 0:
                raise errors.SandboxTaskFailureError("Test tasks failed")

        for side in bans:
            if not verified.get(side):
                raise errors.SandboxTaskFailureError("Too many subsource errors on {} middlesearch".format(side))

        if self._check_removal(verified, removals, rule):
            return
        raise errors.SandboxTaskFailureError("Too many removals; test failure")

    def _check_removal(self, verified, removals, rule):
        for first_task_key in verified.get("first", set()):
            first_removals = removals[first_task_key]
            for second_task_key in verified.get("second", set()):
                removals_delta = float(removals[second_task_key] - first_removals) / (first_removals if first_removals else 1)
                logging.info("rule={}, removals_delta={} for tasks {}, {}".format(rule, removals_delta, first_task_key, second_task_key))
                if removals_delta <= self.ctx[MaxRemovalDeltaParameter.name]:
                    logging.info("Test successful")
                    return True
        return False

    def _get_stats(self, task_id):
        task = channel.sandbox.get_task(task_id)
        return task.ctx.get("stats") or task.ctx.get(metasearch_task.BaseMiddlesearchTask.evlog_stats_key)

    def _get_task_key(self, description, repeat_id, mult_id):
        return "{}_ban_test_{}_{}".format(description, repeat_id, mult_id)

    def _verify_errors(self, stats, message):
        errors_count = stats["subsource_errors"]
        logging.info("message={}, errors_count={}".format(message, errors_count))
        return errors_count <= self.ctx[MaxErrorsCountParameter.name]

    def _test_task(self, ban_resource_id, description, ram_param):
        index_type = media_settings.INDEX_MAIN
        common_cgroup = '{"memory": {"low_limit_in_bytes": "48G", "limit_in_bytes": "49G", "recharge_on_pgfault": "1"}}'
        sub_ctx = {
            metasearch_task.MIDDLESEARCH_PARAMS.Binary.name: self._get_middlesearch_executable(),
            metasearch_task.MIDDLESEARCH_PARAMS.Config.name: self._get_middlesearch_config(),
            metasearch_task.MIDDLESEARCH_PARAMS.Data.name: self._get_middlesearch_data(),
            metasearch_task.MIDDLESEARCH_PARAMS.Cgroup.name: common_cgroup,

            metasearch_task.BASESEARCH_PARAMS.Binary.name: self._get_basesearch_executable(index_type),
            metasearch_task.BASESEARCH_PARAMS.Config.name: self._get_basesearch_config(index_type),
            metasearch_task.BASESEARCH_PARAMS.Cgroup.name: "{}".format(media_settings.ImagesSettings.CGROUPS[index_type]).replace('\'', '\"'),

            metasearch_task.SNIPPETIZER_PARAMS.Binary.name: self._get_basesearch_executable(index_type),
            metasearch_task.SNIPPETIZER_PARAMS.Config.name: self._get_basesearch_config(index_type),
            metasearch_task.SNIPPETIZER_PARAMS.Cgroup.name: "{}".format(media_settings.ImagesSettings.CGROUPS[index_type]).replace('\'', '\"'),

            performance_task.PlanParameter.name: self._get_middlesearch_plan(media_settings.INDEX_MIDDLE, index_type),
            performance_task.CustomBanParameter.name: ban_resource_id,

            dolbilka.DolbilkaExecutorRequestsLimit.name: 8000,
            dolbilka.DolbilkaExecutorMode.name: dolbilka.DolbilkaExecutorMode.FINGER_MODE,
            dolbilka.DolbilkaMaximumSimultaneousRequests.name: 5,
        }
        kwargs = {
            "task_type": performance_task.ImagesTestMiddlesearchPerformance.type,
            "description": "{}, {}".format(self.descr, description),
            "input_parameters": sub_ctx,
            "arch": sandboxapi.ARCH_LINUX,
            "model": "e5-2650",
            "inherit_notifications": True
        }
        if ram_param.isdigit():
            kwargs["ram"] = int(ram_param) * 1024

        sub_task = self.create_subtask(**kwargs)
        return sub_task.id


__Task__ = ImagesTestMiddlesearchBan
