import six
import sys

from sandbox.common.types.client import Tag
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import parameters
from sandbox.projects.common import utils


class YtProxyParameter(parameters.SandboxStringParameter):
    name = "yt_proxy"
    description = "YT proxy"
    default_value = "hahn.yt.yandex.net"
    required = True


class YtTableParameter(parameters.SandboxStringParameter):
    name = "yt_table"
    description = "YT table"
    default_value = "//home/antispam/StaticAntispam/ImagesBans"
    required = True


class UseYtParameter(parameters.SandboxBoolParameter):
    name = "use_yt"
    description = "Use YT (experimental)"
    default_value = False


class YtTask:
    environment = (environments.PipEnvironment("yandex-yt", "0.8.38a1", use_wheel=True),)
    client_tags = Tag.LINUX_PRECISE
    vault_owner = None

    @staticmethod
    def create_params(default_proxy, default_table):
        class _YtProxyParameter(YtProxyParameter):
            default_value = default_proxy

        class _YtTableParameter(YtTableParameter):
            default_value = default_table

        return (UseYtParameter, _YtProxyParameter, _YtTableParameter)

    def _yt_proxy(self):
        return utils.get_or_default(self.ctx, YtProxyParameter)

    def _yt_table(self):
        return utils.get_or_default(self.ctx, YtTableParameter)

    def __yt_client(self):
        import yt.wrapper.client

        proxy = self._yt_proxy()
        token = self.get_vault_data(self.vault_owner, "yt_token")
        return yt.wrapper.client.Yt(proxy=proxy, config={"token": token})

    def _use_yt(self):
        return utils.get_or_default(self.ctx, UseYtParameter)

    def _yt_get(self, columns):
        import yt.wrapper

        client = self.__yt_client()
        table = self._yt_table()

        columns_str = "{" + ",".join(columns) + "}"
        for record in client.read_table(table + columns_str, format=yt.wrapper.JsonFormat()):
            if len(columns) == 1:
                yield self.__ensure_utf8(record[columns[0]])
            else:
                yield [self.__ensure_utf8(record[key]) for key in columns]

    def _yt_check(self):
        client = self.__yt_client()
        table = self._yt_table()
        return client.get_attribute(table, "modification_time")

    @staticmethod
    def __ensure_utf8(x):
        if sys.version_info[0] == 2 and isinstance(x, six.text_type):
            return x.encode("utf-8")
        else:
            return x
