import itertools
import logging
import re
import time

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common import profiling as search_profiling
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.search.basesearch import task as search_task
from sandbox.projects.tank.load_resources import resources as tank_resources
from sandbox.projects.images.basesearch import task as images_task


_EMERGENCY_PREFIX = "emergency-file:"
_RULE_RE = re.compile(r"([a-z0-9.-_]+)\s*(>|<|>=|<=)\s*([0-9]+)%")
_IMPROVEMENT_KEY = "improvement"  # Performance deltas to draw charts in TestEnvironment


class PlanParameter(parameters.ResourceSelector):
    name = 'dolbilo_plan_resource_id'
    description = 'Plan for basesearch'
    resource_type = resource_types.BASESEARCH_PLAN
    group = dolbilka.DOLBILKA_GROUP
    required = True


class CgiParamsParameter(parameters.SandboxStringParameter):
    name = 'cgi_params'
    description = 'Additional cgi parameters (each line for separate experiment)'
    default_value = ''
    multiline = True
    group = dolbilka.DOLBILKA_GROUP


class PerformanceDiffParameter(parameters.SandboxStringParameter):
    name = 'diff_params'
    description = 'Validate diff (each line for separate experiment)'
    default_value = ''
    multiline = True
    group = dolbilka.DOLBILKA_GROUP


class ImagesAnalyzeBasesearchCgiParams(search_performance.NewShootingTask,
                                       search_profiling.ProfilingTask,
                                       images_task.BaseImgsearchTask):
    """
        Analyze basesearch performance with different cgi params
    """

    type = 'IMAGES_ANALYZE_BASESEARCH_CGI_PARAMS'

    input_parameters = \
        (PlanParameter, CgiParamsParameter, PerformanceDiffParameter) + \
        images_task.BaseImgsearchTask.basesearch_input_parameters + \
        search_performance.NewShootingTask.shoot_input_parameters + \
        search_profiling.ProfilingTask.input_parameters

    def on_enqueue(self):
        images_task.BaseImgsearchTask.on_enqueue(self)
        self.execution_space = search_settings.ImagesSettings.basesearch_executable_disk(self.ctx, images_task.BASESEARCH_PARAMS.Database)
        self.required_ram = search_settings.ImagesSettings.basesearch_executable_memory(
            self.ctx,
            images_task.BASESEARCH_PARAMS.Database
        ) + 5 * 1024

    def on_execute(self):
        search_task.BasesearchComponentTask.on_execute(self)

        # shooting
        self._init_virtualenv(tank_resource_type=tank_resources.YANDEX_TANK_VIRTUALENV_19)
        self._dolbilo_shoot(0)
        cgi_variants = self.ctx[CgiParamsParameter.name].split('\n')
        for n, cgi_params in enumerate(cgi_variants, 1):
            self._dolbilo_shoot(n, cgi_params)

        # cross checks
        diff_variants = utils.get_or_default(self.ctx, PerformanceDiffParameter).split('\n')
        diff_message = []
        for n, params in enumerate(itertools.izip_longest(cgi_variants, diff_variants), 1):
            cgi_params, diff_params = params
            message = self._dolbilo_diff(n, cgi_params, diff_params)
            if message:
                diff_message.append(message)

        if diff_message:
            raise errors.SandboxTaskFailureError("Failed checks:\n{}".format("\n".join(diff_message)))

        for i in xrange(1, n + 1):
            self._profiling_diff(
                self.__get_perf_path(0),
                self.__get_perf_path(i),
                self.__get_perf_path(0, i),
                description="0 vs {}".format(i)
            )

    def _get_queries_parameter(self):
        return PlanParameter

    def _dolbilo_shoot(self, n, cgi_params=""):
        basesearch = self._get_imgsearch()
        data_dir = self.__get_perf_path(n)
        variant = self.__get_variant(n, cgi_params)

        if cgi_params and cgi_params.startswith(_EMERGENCY_PREFIX):
            emergency_path = basesearch.get_emergency_file()
            fu.write_file(emergency_path, cgi_params[len(_EMERGENCY_PREFIX):])
            time.sleep(basesearch.get_emergency_check_period() + 1)
            augment_url = None
        else:
            emergency_path = None
            augment_url = cgi_params

        try:
            self._profiling_init(basesearch, data_dir)
            search_performance.NewShootingTask._dolbilo_shoot(
                self,
                basesearch,
                self.ctx[PlanParameter.name],
                variant,
                augment_url=augment_url
            )
            self._profiling_report(basesearch, data_dir, description=variant)
        finally:
            if emergency_path:
                paths.remove_path(emergency_path)

    def _dolbilo_diff(self, n, cgi_params, diff_params):
        if not cgi_params or not diff_params:
            return

        rule = _RULE_RE.match(diff_params)
        if not rule:
            raise errors.SandboxTaskFailureError("Invalid rule format: {}".format(diff_params))

        field_name = rule.group(1)
        stats = self.ctx[self.new_stats_key]
        value1 = stats[self.__get_variant(0, "")][field_name]
        value2 = stats[self.__get_variant(n, cgi_params)][field_name]
        delta = (float(value2 - value1) / value1) * 100 if value1 else 0

        percents = int(rule.group(3))
        operation = rule.group(2)
        if operation == ">":
            result = delta > percents
        elif operation == ">=":
            result = delta >= percents
        elif operation == "<":
            result = delta < percents
        elif operation == "<=":
            result = delta <= percents
        else:
            raise errors.SandboxTaskFailureError("Unsupported operation: {}".format(operation))

        logging.info("rule '{}', delta '{}', percent '{}', result '{}'".format(
            diff_params, delta, percents, result
        ))

        if _IMPROVEMENT_KEY not in self.ctx:
            self.ctx[_IMPROVEMENT_KEY] = delta

        if not result:
            return "rule '{}', delta {}%".format(diff_params, delta)
        else:
            return None

    def __get_variant(self, n, cgi_params=""):
        return "#{}: {}".format(n, cgi_params)

    def __get_perf_path(self, *ns):
        return self.abs_path("perf.{}.data".format("-".join(str(i) for i in ns)))


__Task__ = ImagesAnalyzeBasesearchCgiParams
