import urllib

from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.common import profiling as search_profiling
from sandbox.projects.common import utils
from sandbox.projects.common.differ import coloring
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.search.basesearch import task as search_task
from sandbox.projects.images import sysinfo as images_sysinfo
from sandbox.projects.images.basesearch import components as images_basesearch_components
from sandbox.projects.images.basesearch import task as images_basesearch_task
from sandbox.projects.tank.load_resources import resources as tank_resources

# Used for profile file names
_FIRST_RUN = 0
_SECOND_RUN = 1

_FIRST_VARIANT = "#{}".format(_FIRST_RUN+1)
_SECOND_VARIANT = "#{}".format(_SECOND_RUN+1)
_DIFF_VARIANT = "diff"

BASESEARCH1_PARAMS = images_basesearch_components.create_imgsearch_params(n=1)
BASESEARCH2_PARAMS = images_basesearch_components.create_imgsearch_params(n=2)


class Plan1Parameter(parameters.ResourceSelector):
    name = 'dolbilo_plan1_resource_id'
    description = 'Plan for basesearch #1'
    resource_type = resource_types.BASESEARCH_PLAN
    required = True


class Plan2Parameter(parameters.ResourceSelector):
    name = 'dolbilo_plan2_resource_id'
    description = 'Plan for basesearch #2'
    resource_type = resource_types.BASESEARCH_PLAN
    required = True


# Note: analyze tasks work with half ready plans in acceptance, so ensure_database
# should be enabled in testenv tasks only
class EarlyDatabaseAutoSelectionParameter(parameters.SandboxBoolParameter):
    name = 'early_database_auto_selection'
    description = 'Use early database autoselection'
    default_value = False


def create_images_analyze_basesearch_performance_parameters(basesearch_parameters):
    return (Plan1Parameter, Plan2Parameter, EarlyDatabaseAutoSelectionParameter) + \
        basesearch_parameters + \
        search_task.BasesearchComponentTask.basesearch_input_parameters + \
        search_performance.NewShootingTask.shoot_input_parameters + \
        search_profiling.ProfilingTask.input_parameters


class ImagesAnalyzeBasesearchPerformance(
        images_sysinfo.SysinfoTask,
        search_performance.NewShootingTask,
        search_profiling.ProfilingTask,
        images_basesearch_task.BaseImgsearchTask):
    """
        Analyze basesearch performance
    """

    type = 'IMAGES_ANALYZE_BASESEARCH_PERFORMANCE'

    # Use FusionSearch obj to run rtyserver, used in quick/ultra.
    use_fusion = False

    # run on CUSTOM_IMAGES tagged hosts if possible
    client_tags = (
        search_performance.NewShootingTask.client_tags &
        images_basesearch_task.BaseImgsearchTask.client_tags
    )

    basesearch_parameters = BASESEARCH1_PARAMS.params + BASESEARCH2_PARAMS.params
    input_parameters = create_images_analyze_basesearch_performance_parameters(basesearch_parameters)

    execution_space = 180 * 1024
    required_ram = 96 * 1024

    def initCtx(self):
        self.ctx['kill_timeout'] = 6 * 60 * 60

    @property
    def footer(self):
        footers = []

        performance_footer = search_performance.NewShootingTask.footer.fget(self)
        footers.append({
            'helperName': '',
            'content': performance_footer
        })

        profiling_footer = search_profiling.ProfilingTask.profiling_footer(self)
        if profiling_footer:
            footers.append({
                'helperName': '',
                'content': profiling_footer
            })

        return footers

    def on_enqueue(self):
        images_basesearch_task.BaseImgsearchTask.on_enqueue(self)
        if utils.get_or_default(self.ctx, EarlyDatabaseAutoSelectionParameter):
            self._ensure_search_database()
            self.execution_space = max(
                media_settings.ImagesSettings.basesearch_executable_disk(self.ctx, params.Database)
                for params in (BASESEARCH1_PARAMS, BASESEARCH2_PARAMS)
                )
            self.required_ram = max(
                media_settings.ImagesSettings.basesearch_executable_memory(self.ctx, params.Database)
                for params in (BASESEARCH1_PARAMS, BASESEARCH2_PARAMS)
            ) + 5 * 1024

    def on_execute(self):
        images_basesearch_task.BaseImgsearchTask.on_execute(self)

        if not utils.get_or_default(self.ctx, EarlyDatabaseAutoSelectionParameter):
            self._ensure_search_database()

        self._init_virtualenv(tank_resource_type=tank_resources.YANDEX_TANK_VIRTUALENV_19)
        self._dolbilo_shoot(BASESEARCH1_PARAMS, Plan1Parameter, _FIRST_VARIANT, _FIRST_RUN)
        self._dolbilo_shoot(BASESEARCH2_PARAMS, Plan2Parameter, _SECOND_VARIANT, _SECOND_RUN)

        stats = self.ctx[self.new_stats_key]
        for key in stats[_FIRST_VARIANT]:
            if self._in_stats(key):
                stats.setdefault(_DIFF_VARIANT, {})[key] = search_performance.delta_percent(
                    stats[_FIRST_VARIANT][key],
                    stats[_SECOND_VARIANT][key]
                )
        self._profiling_diff(
            self.__get_perf_path(_FIRST_VARIANT),
            self.__get_perf_path(_SECOND_VARIANT),
            self.__get_perf_path(_DIFF_VARIANT),
            description="diff"
        )

    def _get_queries_parameter(self):
        return None  # Disable database autodetection made by base class

    def get_short_task_result(self):
        if not self.is_completed():
            return None

        return "{:0.2f}%".format(self.ctx[self.new_stats_key][_DIFF_VARIANT]["shooting.rps_0.5"])

    def _dolbilo_shoot(self, basesearch_params, plan_parameter, variant, run_count):
        basesearch = self._get_imgsearch(
            basesearch_params, use_fusion=self.use_fusion, run_count=run_count)
        data_dir = self.__get_perf_path(variant)

        self._profiling_init(basesearch, data_dir)
        search_performance.NewShootingTask._dolbilo_shoot(
            self,
            basesearch,
            self.ctx[plan_parameter.name],
            variant
        )
        self._profiling_report(basesearch, data_dir, description=variant)

    def _format_stats(self, fmt, stats, variant, key):
        if key == "shooting.rps_0.5" and variant == _DIFF_VARIANT:
            return coloring.color_diff(stats[variant][key], max_diff=-1)
        elif key.startswith("shooting.latency_") and variant == _DIFF_VARIANT:
            return coloring.color_diff(stats[variant][key], max_diff=1)
        else:
            txt = search_performance.NewShootingTask._format_stats(self, fmt, stats, variant, key)
            return txt + ("%" if variant == _DIFF_VARIANT else "")

    def _ensure_search_database(self):
        media_settings.ImagesSettings.ensure_search_database(
            self,
            Plan1Parameter,
            BASESEARCH1_PARAMS.Database
        )
        media_settings.ImagesSettings.ensure_search_database(
            self,
            Plan2Parameter,
            BASESEARCH2_PARAMS.Database
        )

    def __get_perf_path(self, variant):
        return self.abs_path("perf.{}.data".format(urllib.quote(variant, safe="")))


__Task__ = ImagesAnalyzeBasesearchPerformance
