import logging

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types
from sandbox.projects.images.basesearch import ImagesAnalyzeBasesearchPerformance as basesearch_task
from sandbox.projects.images.daemons import ImagesAnalyzeCbirdaemon2Performance as cbirdaemon_task
from sandbox.projects.images.daemons import ImagesAnalyzeNaildaemonPerformance as naildaemon_task
from sandbox.projects.images.daemons import ImagesAnalyzeRimdaemonPerformance as rimdaemon_task


_COMPATIBLE_TASK_TYPES = [
    basesearch_task.ImagesAnalyzeBasesearchPerformance.type,
    cbirdaemon_task.ImagesAnalyzeCbirdaemon2Performance.type,
    naildaemon_task.ImagesAnalyzeNaildaemonPerformance.type,
    rimdaemon_task.ImagesAnalyzeRimdaemonPerformance.type,
]
_DIFF_KEY = "has_diff"
_DELTAS_KEY = "deltas"


class Task1Parameter(parameters.TaskSelector):
    name = 'task1_id'
    description = 'Task 1'
    task_type = _COMPATIBLE_TASK_TYPES


class Task2Parameter(parameters.TaskSelector):
    name = 'task2_id'
    description = 'Task 2'
    task_type = _COMPATIBLE_TASK_TYPES


class MaxDeltaParameter(parameters.SandboxIntegerParameter):
    name = 'max_delta'
    description = 'Maximum possible delta'
    default_value = 2


class ComparedValueNameParameter(parameters.SandboxStringParameter):
    name = 'compared_value'
    description = 'Name of compared value'
    default_value = 'shooting.rps_0.5'


class ImagesCompareAnalyzePerformance(task.SandboxTask):
    """
        Compare results of two performance tasks
    """

    type = 'IMAGES_COMPARE_ANALYZE_PERFORMANCE'

    input_parameters = (Task1Parameter, Task2Parameter, MaxDeltaParameter, ComparedValueNameParameter)

    @property
    def footer(self):
        if _DELTAS_KEY not in self.ctx or _DIFF_KEY not in self.ctx:
            return {"Calculating": "&nbsp;"}

        deltas = self.ctx[_DELTAS_KEY]
        return {"Delta1": deltas[0], "Delta2": deltas[1], "Has diff": self.ctx[_DIFF_KEY]}

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        self._create_resource(
            self.descr,
            self.__get_result_path(),
            resource_types.BASESEARCH_PERFORMANCE_COMPARE_RESULT
        )

    def on_execute(self):
        deltas = [self.__get_delta(parameter, self.ctx[ComparedValueNameParameter.name]) for parameter in (Task1Parameter, Task2Parameter)]
        self.ctx[_DELTAS_KEY] = deltas
        self.ctx[_DIFF_KEY] = abs(deltas[0] - deltas[1]) > self.ctx[MaxDeltaParameter.name]
        with open(self.__get_result_path(), "w") as result_file:
            result_file.write("Delta1: {}, Delta2: {}\n".format(deltas[0], deltas[1]))

    def __get_delta(self, parameter, name):
        task_input = channel.channel.sandbox.get_task(self.ctx[parameter.name])
        diff_ctx = task_input.ctx.get("new_stats", {}).get('diff', {})
        if name in diff_ctx:
            return diff_ctx[name]
        logging.info('Subtask #{0} ctx: {1}'.format(self.ctx[parameter.name], str(task_input.ctx)))
        return None

    def __get_result_path(self):
        return self.abs_path("diff.txt")


__Task__ = ImagesCompareAnalyzePerformance
