import base64
import logging
import os
import random
import re
import time
import urllib

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils

from sandbox.projects.common.base_search_quality import response_saver

from sandbox.projects.common.search import queries as search_queries
from sandbox.projects.common.search import requester_compat as search_requester
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.images.basesearch import task as images_task


_RESPONSE_SAVER_PARAMS = response_saver.create_response_saver_params()

# Don't randomize following params to preserve query type
_QUERY_TYPE_CGI_PARAMS = set(["haha", "DF"])


class InitialSeedParameter(parameters.SandboxIntegerParameter):
    name = 'initial_seed'
    description = 'Initial random seed (use 0 for current time)'
    group = "Task parameters"
    default_value = 0


class ImagesTestBasesearchRandom(images_task.BaseImgsearchTask):
    type = 'IMAGES_TEST_BASESEARCH_RANDOM'

    input_parameters = \
            (InitialSeedParameter,) + \
            images_task.BaseImgsearchTask.basesearch_input_parameters + \
            _RESPONSE_SAVER_PARAMS.params + \
            search_requester.create_params()

    def on_enqueue(self):
        images_task.BaseImgsearchTask.on_enqueue(self)
        self.execution_space = search_settings.ImagesSettings.basesearch_executable_disk(self.ctx, images_task.BASESEARCH_PARAMS.Database)
        self.required_ram = search_settings.ImagesSettings.basesearch_executable_memory(
            self.ctx,
            images_task.BASESEARCH_PARAMS.Database
        ) + 5 * 1024

    def on_execute(self):
        images_task.BaseImgsearchTask.on_execute(self)
        basesearch = self._get_imgsearch()
        with basesearch:
            queries_path = self.sync_resource(self.ctx[self._get_queries_parameter().name])
            basesearch.use_component(lambda: self.__save_responses(queries_path, basesearch))

    def _get_queries_parameter(self):
        return _RESPONSE_SAVER_PARAMS.QueriesParameter

    def __save_responses(self, queries_path, basesearch):
        eh.ensure(os.stat(queries_path).st_size, "Input file with queries is empty")

        requests_iterator = self.__randomize_queries(queries_path, basesearch)
        for response_num, response_status, response_data in search_requester.save_responses(self.ctx, requests_iterator):
            pass  # Ignore response errors

        if self.__generator_error:
            raise errors.SandboxTaskFailureError("Error in query generator: {}".format(self.__generator_error))

    def __randomize_queries(self, queries_path, basesearch):
        initial_seed = utils.get_or_default(self.ctx, InitialSeedParameter)
        if initial_seed == InitialSeedParameter.default_value:
            self.ctx[InitialSeedParameter.name] = int(time.time())

        self.__generator_error = None
        for query in search_requester.generate_queries_old(self.ctx, queries_path, basesearch):
            try:
                query_params = search_queries.parse_cgi_params(query.split("?")[1], unquote=False)

                # Initialize random generator
                reqid = query_params.get("reqid", [""])[0]
                seed = hash(reqid) + initial_seed
                random_gen = random.Random(seed)

                # Select input cgi parameter and randomize it
                choice_params = [
                    (key, value)
                    for key, values in query_params.iteritems() for value in values
                    if key not in _QUERY_TYPE_CGI_PARAMS and value
                ]
                random_param = random_gen.choice(choice_params)
                random_len = random_gen.randrange(100)
                random_value = "".join(chr(random_gen.choice(range(256))) for _ in xrange(random_len))
                if random_gen.choice((True, False)):
                    random_value = base64.b64encode(random_value)
                if random_param[0] == "relev":
                    random_relev = random_gen.choice([v for v in random_param[1].split(";") if "%3D" in v])
                    random_value = urllib.quote(random_value.replace(";", ""))
                    random_value = re.sub(r"(^|;){}($|;)".format(random_relev),
                                          r"\1{}%3D{}\2".format(random_relev.split("%3D")[0], random_value),
                                          random_param[1], count=1)
                    logging.info("Randomizing relev {} {} -> {}".format(random_relev, random_param[1], random_value))
                elif random_param[0] == "pron":
                    random_pos = random_gen.randint(0, len(random_param[1]))
                    random_value = urllib.quote(random_param[1][:random_pos] + random_value)
                    logging.info("Randomizing pron {} {} -> {}".format(random_pos, random_param[1], random_value))
                else:
                    random_value = urllib.quote(random_value)
                    logging.info("Randomizing {} {} -> {}".format(random_param[0], random_param[1], random_value))

                old_param = re.escape("{}={}".format(random_param[0], random_param[1]))
                new_param = "{}={}".format(random_param[0], random_value)
                new_query = re.sub(r"(^|&){}($|&)".format(old_param), r"\1{}\2".format(new_param), query, count=1)
                yield new_query
            except Exception as e:
                logging.error("Problems with request: {}".format(e))
                self.__generator_error = e
                break


__Task__ = ImagesTestBasesearchRandom
