import json
import logging

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox import sandboxsdk
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import utils
from sandbox.projects.common import cgroup as cgroup_api
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search import settings as search_settings


def create_imgsearch_params(use_fusion=False, **kwargs):
    if use_fusion:
        return create_quick_imgsearch_params(**kwargs)

    return search_components.create_basesearch_params(
        database_required=False,
        default_patch_request_threads=False,
        archive_model_required=False,
        default_start_timeout=1200,
        **kwargs
    )


def create_quick_imgsearch_params(**kwargs):
    return search_components.create_images_fusion_params(**kwargs)


def get_quick_imgsearch(params=create_quick_imgsearch_params(), **kwargs):
    return search_components.get_fusion_search(
        params,
        use_frozen_configs=True,
        extra_params=_get_default_fusion_extra_params(),
        **kwargs
    )


def get_imgsearch(params=create_imgsearch_params(), *args, **kwargs):
    task = channel.task
    index_type = search_settings.ImagesSettings.basesearch_database_index_type(task.ctx, params.Database)

    return search_components.get_basesearch(
        params=params,
        cgroup=_create_basesearch_cgroup(index_type, utils.get_or_default(task.ctx, params.Cgroup)),
        *args,
        **kwargs
    )


def _create_basesearch_cgroup(index_type, cgroup_props):
    if common.config.Registry().common.installation == ctm.Installation.LOCAL:
        logging.info("Skipping cgroups creation on local sandbox")
        return None

    if cgroup_props:
        cgroup_props = json.loads(cgroup_props)
    else:
        cgroup_props = search_settings.ImagesSettings.CGROUPS[index_type]

    return cgroup_api.create_cgroup(index_type, cgroup_props)


def _get_default_fusion_extra_params():
    detach_path = sandboxsdk.paths.make_folder("detach")
    state_path = sandboxsdk.paths.make_folder("state")

    return {
        "DETACH_DIRECTORY": detach_path,
        "STATE_ROOT": state_path,
        "BSCONFIG_INAME": "refresh",
        "CPU_LIMIT": "8",
        "CTYPE": "integrations_tests",
        "ThreadsMult": "1",
    }
