from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images.basesearch import resources as basesearch_resources
from sandbox.projects.images.models import resources as models_resources
from sandbox.projects.images import util
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

BASESEARCH_DEFAULT_PORT = 17171


class BaseSearchPartition:
    """
        Abstract task for images basesearch running
    """

    basesearch_databases = []

    class Context(sdk2.Task.Context):
        __first_run = True
        __shards_search = False
        basesearch_databases_ctx = []

    class Parameters(sdk2.Task.Parameters):
        # Basesearch executable
        basesearch_executable = sdk2.parameters.Resource('Basesearch executable',
                                                         resource_type=basesearch_resources.IMGSEARCH_EXECUTABLE,
                                                         required=False)

        # Matrixnet models archive
        basesearch_models_archive = sdk2.parameters.Resource('Basesearch matrixnet models archive',
                                                             resource_type=models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE,
                                                             required=False)

        # Basesearch shard number
        basesearch_shard_numbers = sdk2.parameters.List('Basesearch shards number',
                                                        value_type=sdk2.parameters.Integer,
                                                        default=[0],
                                                        required=False)

        # Database
        basesearch_databases = sdk2.parameters.List('Basesearch databases: use it for manually builded shards',
                                                    value_type=sdk2.parameters.Integer,
                                                    required=False)

        # Config
        basesearch_config = sdk2.parameters.Resource('Basesearch config: use it for compatibility with old sdk1 tasks',
                                                     required=False)

    def on_enqueue(self):
        if self.Context.__first_run is False:
            return

        if self.Parameters.basesearch_executable is None:
            self.Parameters.basesearch_executable = sdk2.Resource.find(type=basesearch_resources.IMGSEARCH_EXECUTABLE,
                                                                       attrs=dict(released='stable'),
                                                                       state='READY').first()
        if self.Parameters.basesearch_models_archive is None:
            self.Parameters.basesearch_models_archive = sdk2.Resource.find(type=models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE,
                                                                           attrs=dict(released='stable'),
                                                                           state='READY').first()
        if self.Parameters.basesearch_config is None:
            self.Parameters.basesearch_config = sdk2.Resource.find(type=basesearch_resources.IMAGES_SEARCH_CONFIG,
                                                                   attrs=dict(released='stable'),
                                                                   state='READY').first()

        if not self.Parameters.basesearch_databases or self.Parameters.basesearch_databases == [0]*len(self.Parameters.basesearch_shard_numbers):
            shards = []
            self.Context.__shards_search = True
            for i in self.Parameters.basesearch_shard_numbers:
                shard_name = BaseSearchPartition.get_shard_name(self, self.Parameters.index_state, i)
                resource = sdk2.Resource.find(type=resource_types.IMAGES_SEARCH_DATABASE,
                                              attrs=dict(shard_instance=shard_name),
                                              state='READY').first()
                if resource is None:
                    shards.append(0)
                else:
                    shards.append(resource.id)
            self.Parameters.basesearch_databases = shards

        shards = []
        for i in range(len(self.Parameters.basesearch_shard_numbers)):
            if self.Parameters.basesearch_databases[i] == 0:
                shard_name = BaseSearchPartition.get_shard_name(self, self.parameters.index_state, self.Parameters.basesearch_shard_numbers[i])
                shards.append(("shard", (media_settings.INDEX_MAIN, shard_name)))
            else:
                shards.append(("resource", self.Parameters.basesearch_databases[i]))
        self.Context.basesearch_databases_ctx = shards

        self.Context.__first_run = False

    def init_resources(self):
        if self.Parameters.basesearch_executable is None:
            raise SandboxTaskFailureError("Released stable \"Basesearch executable\" not found")
        if self.Parameters.basesearch_models_archive is None:
            raise SandboxTaskFailureError("Released stable \"Basesearch matrixnet models archive\" not found")
        if self.Parameters.basesearch_config is None:
            raise SandboxTaskFailureError("Released stable \"Basesearch config\" not found")

        self.Context.basesearch_databases_ctx = map(lambda x: util.images_database_search(self, x), self.Context.basesearch_databases_ctx)
        self.basesearch_databases = util.images_database_get_resources(self.Context.basesearch_databases_ctx)

    def get_shard_name(self, index_state, shard_number):
        return "imgsidx-{:03d}-{}".format(shard_number, index_state)
