from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images.basesearch import resources as basesearch_resources
from sandbox.projects.images.models import resources as models_resources
from sandbox.projects.images import util
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

BASESEARCH_DEFAULT_PORT = 17171


class BaseSearchTask:
    """
        Abstract task for images basesearch running
    """

    basesearch_database = None

    class Context(sdk2.Task.Context):
        basesearch_first_run = True
        basesearch_database_ctx = None

    class Parameters(sdk2.Task.Parameters):
        # Basesearch executable
        basesearch_executable = sdk2.parameters.Resource('Basesearch executable',
                                                         resource_type=basesearch_resources.IMGSEARCH_EXECUTABLE,
                                                         required=False)

        # Matrixnet models archive
        basesearch_models_archive = sdk2.parameters.Resource('Basesearch matrixnet models archive',
                                                             resource_type=models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE,
                                                             required=False)

        # Basesearch shard number
        basesearch_shard_number = sdk2.parameters.Integer('Basesearch shard number',
                                                          default=0,
                                                          required=False)

        # Database
        basesearch_database = sdk2.parameters.Resource('Basesearch database: use it for manually builded shards',
                                                       resource_type=resource_types.IMAGES_SEARCH_DATABASE,
                                                       required=False)

        # Config
        basesearch_config = sdk2.parameters.Resource('Basesearch config: use it for compatibility with old sdk1 tasks',
                                                     required=False)

    def on_enqueue(self):
        if self.Context.basesearch_first_run is False:
            return

        shard_name = BaseSearchTask.get_shard_name(self, self.Parameters.index_state, self.Parameters.partition_number)

        if self.Parameters.basesearch_executable is None:
            self.Parameters.basesearch_executable = sdk2.Resource.find(type=basesearch_resources.IMGSEARCH_EXECUTABLE,
                                                                       attrs=dict(released='stable'),
                                                                       state='READY').first()
        if self.Parameters.basesearch_models_archive is None:
            self.Parameters.basesearch_models_archive = sdk2.Resource.find(type=models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE,
                                                                           attrs=dict(released='stable'),
                                                                           state='READY').first()
        if self.Parameters.basesearch_config is None:
            self.Parameters.basesearch_config = sdk2.Resource.find(type=basesearch_resources.IMAGES_SEARCH_CONFIG,
                                                                   attrs=dict(released='stable'),
                                                                   state='READY').first()
        if self.Parameters.basesearch_database is None:
            self.Parameters.basesearch_database = sdk2.Resource.find(type=resource_types.IMAGES_SEARCH_DATABASE,
                                                                     attrs=dict(shard_instance=shard_name),
                                                                     state='READY').first()

        if self.Parameters.basesearch_database is None:
            self.Context.basesearch_database_ctx = ("shard", (media_settings.INDEX_MAIN, shard_name))
        else:
            self.Context.basesearch_database_ctx = ("resource", self.Parameters.basesearch_database.id)

        self.Context.basesearch_first_run = False

    def init_resources(self):
        if self.Parameters.basesearch_executable is None:
            raise SandboxTaskFailureError("Released stable \"Basesearch executable\" not found")
        if self.Parameters.basesearch_models_archive is None:
            raise SandboxTaskFailureError("Released stable \"Basesearch matrixnet models archive\" not found")
        if self.Parameters.basesearch_config is None:
            raise SandboxTaskFailureError("Released stable \"Basesearch config\" not found")

        self.Context.basesearch_database_ctx = util.images_database_search(self, self.Context.basesearch_database_ctx)

        self.basesearch_database = util.images_database_get_resources([self.Context.basesearch_database_ctx])[0]

    def get_shard_name(self, index_state, shard_number):
        return "imgsidx-{:03d}-{}".format(shard_number, index_state)
