import urllib

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types
from sandbox.projects.common import profiling as search_profiling
from sandbox.projects.common.differ import coloring
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.images.daemons import components as daemons_components
from sandbox.projects.tank.load_resources import resources as tank_resources

RIMDAEMON1_PARAMS = daemons_components.create_rimdaemon_params(n=1)
RIMDAEMON2_PARAMS = daemons_components.create_rimdaemon_params(n=2)

DIFF_VARIANT = "diff"


class Plan1Parameter(parameters.ResourceSelector):
    name = 'dolbilo_plan1_resource_id'
    description = 'Plan for rimdaemon #1'
    resource_type = resource_types.BASESEARCH_PLAN
    required = True


class Plan2Parameter(parameters.ResourceSelector):
    name = 'dolbilo_plan2_resource_id'
    description = 'Plan for rimdaemon #2'
    resource_type = resource_types.BASESEARCH_PLAN
    required = True


class ImagesAnalyzeRimdaemonPerformance(search_performance.NewShootingTask,
                                        search_profiling.ProfilingTask,
                                        task.SandboxTask):
    """
        Analyze rimdaemon performance
    """

    type = 'IMAGES_ANALYZE_RIMDAEMON_PERFORMANCE'
    execution_space = 70 * 1024  # 70 GB

    input_parameters = \
        (Plan1Parameter, Plan2Parameter) + \
        RIMDAEMON1_PARAMS.params + \
        RIMDAEMON2_PARAMS.params + \
        search_performance.NewShootingTask.shoot_input_parameters + \
        search_profiling.ProfilingTask.input_parameters

    @property
    def footer(self):
        footers = []

        performance_footer = search_performance.NewShootingTask.footer.fget(self)
        footers.append({
            'helperName': '',
            'content': performance_footer
        })

        profiling_footer = search_profiling.ProfilingTask.profiling_footer(self)
        if profiling_footer:
            footers.append({
                'helperName': '',
                'content': profiling_footer
            })

        return footers

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        media_settings.ImagesSettings.ensure_search_database(
            self,
            Plan1Parameter,
            RIMDAEMON1_PARAMS.Database
        )
        media_settings.ImagesSettings.ensure_search_database(
            self,
            Plan2Parameter,
            RIMDAEMON2_PARAMS.Database
        )

    def on_execute(self):
        self._init_virtualenv(tank_resource_type=tank_resources.YANDEX_TANK_VIRTUALENV_19)
        self._dolbilo_shoot(RIMDAEMON1_PARAMS, Plan1Parameter, 1)
        self._dolbilo_shoot(RIMDAEMON2_PARAMS, Plan2Parameter, 2)

        stats = self.ctx[self.new_stats_key]
        for key in stats["#1"]:
            if self._in_stats(key):
                stats.setdefault(DIFF_VARIANT, {})[key] = _delta(stats["#1"][key], stats["#2"][key])
        self._profiling_diff(
            self.__get_perf_path("#1"),
            self.__get_perf_path("#2"),
            self.__get_perf_path(DIFF_VARIANT),
            description="diff"
        )

    def get_short_task_result(self):
        if not self.is_completed():
            return None

        return "{:0.2f}%".format(self.ctx[self.new_stats_key][DIFF_VARIANT]["shooting.rps_0.5"])

    def _dolbilo_shoot(self, rimdaemon_params, plan_parameter, run_count):
        variant = "#{}".format(run_count)

        rimdaemon = daemons_components.get_rimdaemon(rimdaemon_params)
        data_dir = self.__get_perf_path(variant)

        self._profiling_init(rimdaemon, data_dir)
        search_performance.NewShootingTask._dolbilo_shoot(
            self,
            rimdaemon,
            self.ctx[plan_parameter.name],
            variant
        )
        self._profiling_report(rimdaemon, data_dir, description=variant)

    def _format_stats(self, fmt, stats, variant, key):
        if key == "shooting.rps_0.5" and variant == DIFF_VARIANT:
            return coloring.color_diff(stats[variant][key], max_diff=-1)
        elif key.startswith("shooting.latency_") and variant == DIFF_VARIANT:
            return coloring.color_diff(stats[variant][key], max_diff=1)
        else:
            txt = search_performance.NewShootingTask._format_stats(self, fmt, stats, variant, key)
            return txt + ("%" if variant == DIFF_VARIANT else "")

    def __get_perf_path(self, variant):
        return self.abs_path("perf.{}.data".format(urllib.quote(variant, safe="")))


def _delta(a, b):
    return (float(b - a) / a) * 100 if a else 0


__Task__ = ImagesAnalyzeRimdaemonPerformance
