# -*- coding: utf-8 -*-

import datetime
import logging
import os.path
import shutil
import time

from sandbox import sdk2
import sandbox.sdk2.helpers as sdk2_helpers

import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.common.decorators import retries
from sandbox.projects.common import decorators
from sandbox.projects.common.nanny import nanny
from sandbox.projects.images.daemons import resources as daemons_resources
from sandbox.projects.images.deployment.ImagesSwitchBinary import ImagesSwitchBinary

from sandbox.projects.MediaLib.media_zk import MediaZkClient

RETRIES = 10
DELAY = 30

SUPPORTED_SHARDS = [
    daemons_resources.CBIR_DAEMON2_SETTINGS,
    daemons_resources.CBIR_DAEMON2_API_SETTINGS,
    daemons_resources.CBIR_DAEMON2_GPU_SETTINGS,
    daemons_resources.CBIR_DAEMON2_GPU_API_SETTINGS,
]


class ArcadiaUrlParameter(sdk2.parameters.ArcadiaUrl):
    default_value = 'arcadia:/arc/trunk/arcadia'

class ArcadiaPathParameter(sdk2.parameters.String):
    default_value = 'extsearch/images/daemons/cbirdaemon2/data'

class ImagesBuildCbirdaemon2Database(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Build bundle with cbirdaemon data
    """
    DEPLOY_GROUP = 'IMAGES-BASE-DEPLOY'

    DEPLOY_CC_USERS = [
    ]

    DEPLOY_QUEUE_ID = "IMAGES"

    DEPLOY_DASHBOARD_LINK_FMT = \
        "https://nanny.yandex-team.ru/ui/#/services/dashboards/catalog/{dashboard}/recipes/catalog/{recipe}/"

    CIRCUITS = []

    DEPLOY_TASK_NAME = 'IMAGES_SWITCH_BINARY'

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = ArcadiaUrlParameter('Svn url for arcadia', required=True)
        arcadia_shards_dir = ArcadiaPathParameter('Shard packages dir', required=True)
        arcadia_patch = sdk2.parameters.String(
            "Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4",
            multiline=True,
            default="",
        )
        with sdk2.parameters.CheckGroup("Build shard(s)", required=True) as shards_to_build:
            for resource in SUPPORTED_SHARDS:
                resource_name = str(resource.name)
                setattr(shards_to_build.values,
                        resource_name,
                        shards_to_build.Value(resource_name, checked=True)
                )

    class Context(sdk2.Task.Context):
        release_changelog = u'empty description'
        nanny_ticket = None
        shard_id = u'undefined'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        disk_space = 50 * 1024  # 50 Gb

    def get_resource(self, resource_type_name):
        for resource in SUPPORTED_SHARDS:
            if str(resource.name) == resource_type_name:
                return resource
        return None

    def build_shard(self, arcadia_data_dir, shard_id, resource):
        database_path = "{}-{}".format(
            resource.shard_instance_prefix,
            shard_id
        )
        paths.make_folder(database_path, delete_content=True)
        for subdir in paths.list_dir(arcadia_data_dir, folders_only=True, abs_path=True):
            for fname in paths.list_dir(subdir):
                abs_fname = os.path.join(subdir, fname)
                if os.path.isfile(abs_fname):
                    shutil.copy(abs_fname, database_path)
                else:
                    dst = os.path.join(database_path, fname)
                    paths.copytree3(abs_fname, dst)
        self.__shard_cmd("configure", database_path)
        self.__shard_cmd("register", "--with-dir", database_path)
        cbirdaemon_settings = resource(
            self,
            self.Parameters.description,
            database_path,
            shard_instance=database_path
        )
        sdk2.ResourceData(cbirdaemon_settings).ready()

    def on_execute(self):
        with self.memoize_stage.build(commit_on_entrance=False):
            shard_id = datetime.datetime.now().strftime("%Y%m%d-%H%M%S-%f")
            self.Context.shard_id = shard_id
            build_results = 'build-{}'.format(shard_id)
            paths.make_folder(build_results, delete_content=True)
            build_targets = [
                os.path.join(self.Parameters.arcadia_shards_dir, self.get_resource(shard_name).arcadia_dir)
                for shard_name in self.Parameters.shards_to_build
            ]

            arcadia_root = arcadiasdk.do_clone(self.Parameters.checkout_arcadia_from_url, self)
            arcadia_src_dir = os.path.join(arcadia_root, self.Parameters.arcadia_shards_dir)
            arcadia_shards_url = sdk2.svn.Arcadia.append(self.Parameters.checkout_arcadia_from_url,
                                                         self.Parameters.arcadia_shards_dir)
            sdk2.svn.Arcadia.checkout(arcadia_shards_url, arcadia_src_dir)
            if self.Parameters.arcadia_patch:
                arcadiasdk.apply_patch(self, arcadia_root, self.Parameters.arcadia_patch, self.path())
            arcadiasdk.do_build(
                build_system=sdk_constants.YMAKE_BUILD_SYSTEM,
                source_root=arcadia_root,
                targets=build_targets,
                results_dir=build_results,
                checkout=True,
                clear_build=True,
            )
            for i, shard_name in enumerate(self.Parameters.shards_to_build):
                self.build_shard(os.path.join(build_results, build_targets[i], self.Parameters.arcadia_shards_dir),
                                 shard_id,
                                 self.get_resource(shard_name)
                )

    def __shard_cmd(self, *args):
        with sdk2_helpers.ProcessLog(self, logger=logging.getLogger("iss_shards")) as pl:
            sdk2_helpers.subprocess.check_call(
                (self.__shard_tool(),) + args,
                stdout=pl.stdout,
                stderr=sdk2_helpers.subprocess.STDOUT
            )

    @decorators.memoize
    def __shard_tool(self):
        tool_resource = resource_types.ISS_SHARDS.find(attrs={"released": sandboxapi.RELEASE_STABLE}).first()
        if not tool_resource:
            raise errors.SandboxTaskFailureError("Failed to find last released {}".format(resource_types.ISS_SHARDS))
        return str(sdk2.ResourceData(tool_resource).path)

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _create_nanny_ticket(self, nanny_client, state_name):
        title='Переключение данных сибирь-демонов на стейт {}'.format(state_name)
        if self.Context.release_changelog is ctm.NotExists:
            self.Context.release_changelog = u'empty description'
        release_changelog = self.Context.release_changelog
        return nanny_client.create_ticket(
            queue_id='{}'.format(self.DEPLOY_QUEUE_ID),
            title=title,
            description=u'{}'.format(release_changelog),
            responsible='woxalex',
            urgent=True,
            copy_to=self.DEPLOY_CC_USERS
        )

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _update_nanny_ticket(self, nanny_client, text):
        nanny_client.update_ticket_status(self.Context.nanny_ticket, 'IN_QUEUE', text)

    def _create_deploy_ticket(self):
        if self.Context.shard_id is ctm.NotExists:
            self.Context.shard_id = 'undefined'

        # create ticket
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=sdk2.Vault.data('MEDIA_DEPLOY', 'nanny-oauth-token')
        )
        nanny_ticket = self._create_nanny_ticket(nanny_client, self.Context.shard_id)
        self.Context.nanny_ticket = nanny_ticket['value']['id']
        logging.debug("Created Nanny ticket: {}".format(self.Context.nanny_ticket))
        self.set_info("Created Nanny ticket: <a href=https://nanny.yandex-team.ru/ui/#/t/{ticket}/>{ticket}</a>".format(
            ticket=self.Context.nanny_ticket),
            do_escape=False)

        zk_sync_node = ImagesSwitchBinary.ZK_SYNC_NODE + '_cbirdaemon'
        self.CIRCUITS = [
            {
                'description': u'Переключение шарда сибирь-демонов',
                'dashboard': 'images-cbirdaemon',
                'recipe': 'cbrd_deploy_database',
                'zk_lock': '/media-services/images/flags/deployment/production_sas_imgcbrd',
                'semaphore': 'IMAGES_CBRD_DEPLOY'
            }
        ]

        # create task in draft status
        zk_value = str(len(self.CIRCUITS))
        with MediaZkClient() as zk:
            if zk.exists(zk_sync_node):
                zk.set(zk_sync_node, zk_value)
            else:
                zk.create(zk_sync_node, zk_value, makepath=True)
        for circuit in self.CIRCUITS:
            input_ctx = {
                ImagesSwitchBinary.Parameters.release_task.name: self.id,
                ImagesSwitchBinary.Parameters.nanny_dashboard_name.name: circuit['dashboard'],
                ImagesSwitchBinary.Parameters.nanny_dashboard_recipe.name: circuit['recipe'],
                ImagesSwitchBinary.Parameters.semaphore_name.name: circuit['semaphore'],
                ImagesSwitchBinary.Parameters.zookeeper_path.name: circuit['zk_lock'],
                ImagesSwitchBinary.Parameters.nanny_ticket.name: self.Context.nanny_ticket,
                ImagesSwitchBinary.Parameters.zk_sync_node.name: zk_sync_node
            }
            switch_task = sdk2.Task[self.DEPLOY_TASK_NAME]
            sandbox_task_id = switch_task(
                switch_task.current,
                description="Deploy cbirshard {} to production services, dashboard {} recipe {}".format(self.Context.shard_id, circuit['dashboard'], circuit['recipe']),
                owner=self.DEPLOY_GROUP,
                priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
                **input_ctx).id

            # set comment with link to sandbox task
            dashboard_link = self.DEPLOY_DASHBOARD_LINK_FMT.format(dashboard=circuit['dashboard'],
                                                                   recipe=circuit['recipe'])
            text = u'''{}: https://sandbox.yandex-team.ru/task/{}/
Метарецепт: {}
'''.format(circuit['description'], sandbox_task_id, unicode(dashboard_link, 'utf-8'))
            self._update_nanny_ticket(nanny_client, text)

    def on_release(self, additional_parameters):
        if additional_parameters['release_status'] == ctt.ReleaseStatus.STABLE:
            try:
                self._create_deploy_ticket()
            except Exception as err:
                logger = logging.getLogger()
                logger.exception(err)
                self.set_info("<strong>Can't create deploy ticket.</strong>", do_escape=False)
        # Ticket integration logic
        # ========================
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
