import itertools
import os.path

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task

import sandbox.projects.common.app_host.converter as apphost_converter
from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common.search import requester as search_requester
from sandbox.projects.common.differ import printers
from sandbox.projects.images.daemons import resources as images_daemons_resources

import compare_responses

_DIFF_COUNT_KEY = "diff_count"
_COMPARE_RESULT_KEY = "compare_result"


class Responses1Parameter(parameters.ResourceSelector):
    name = "cbir_daemon_responses1_resource_id"
    description = "Cbirdaemon responses #1"
    resource_type = images_daemons_resources.CBIR_DAEMON2_APPHOST_RESPONSES


class Responses2Parameter(parameters.ResourceSelector):
    name = "cbir_daemon_responses2_resource_id"
    description = "Cbirdaemon responses #2"
    resource_type = images_daemons_resources.CBIR_DAEMON2_APPHOST_RESPONSES


class ImagesCompareCbirdaemon2ApphostResponses(task.SandboxTask):
    """
        Get cbirdaemon responses (apphost mode)
    """

    type = 'IMAGES_COMPARE_CBIRDAEMON2_APPHOST_RESPONSES'

    input_parameters = (Responses1Parameter, Responses2Parameter)

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        self.create_resource(
            self.descr,
            self.__get_diff_dir(),
            images_daemons_resources.CBIR_DAEMON2_APPHOST_RESPONSES_DIFF
        )

    def on_execute(self):
        diff_path = self.__get_diff_dir()
        paths.make_folder(diff_path, delete_content=True)

        iterator1 = self.__responses_iterator(self.ctx[Responses1Parameter.name])
        iterator2 = self.__responses_iterator(self.ctx[Responses2Parameter.name])

        diff_content = []
        for n, responses in enumerate(itertools.izip_longest(iterator1, iterator2)):
            response1, response2 = responses
            if response1 is None or response2 is None:
                raise errors.SandboxTaskFailureError("Different number of responses")

            output = compare_responses.compare(response1, response2, "response #{}".format(n), diff_path)
            if output:
                self.ctx[_DIFF_COUNT_KEY] = self.ctx.get(_DIFF_COUNT_KEY, 0) + 1
                diff_content.extend(output)

        if diff_content:
            printers.HtmlFile.write_companion(diff_path, "response_diff_style.css")
            printers.HtmlFile.write_companion(diff_path, "response_diff_scripts.js")
            with open(os.path.join(diff_path, "diff.html"), "w") as diff_file:
                printers.HtmlFile.write(diff_file, "".join(diff_content), buttons=False)
            self.ctx["compare_result"] = False
        else:
            fu.write_file(os.path.join(diff_path, utils.NO_DIFF_NAME), utils.no_diff_detected())
            self.ctx["compare_result"] = True

    def get_short_task_result(self):
        if _COMPARE_RESULT_KEY in self.ctx:
            return "no diff" if self.ctx[_COMPARE_RESULT_KEY] else "{} diff".format(self.ctx[_DIFF_COUNT_KEY])

    def __responses_iterator(self, responses_resource_id):
        responses_path = self.sync_resource(responses_resource_id)
        # TODO: don't use converter released with noapache
        converter_path = self.sync_resource(utils.get_and_check_last_released_resource_id(
            resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE,
            arch=sandboxapi.ARCH_LINUX
        ))
        converter_func = apphost_converter.convert_input_response
        with open(responses_path) as responses_file:
            for data in search_requester.sequence_binary_data_from_stream(responses_file):
                yield converter_func(converter_path, data)

    def __get_diff_dir(self):
        return self.abs_path("diff")


__Task__ = ImagesCompareCbirdaemon2ApphostResponses
