import base64
import itertools
import os
import json

from sandbox.sandboxsdk import errors
from sandbox.projects.common.differ import printers


EPS_BY_PATH = {
    "/answers/*/signatures/similarnn/crop_features/*/rect/x": 1.0,
    "/answers/*/signatures/similarnn/crop_features/*/rect/y": 1.0,
    "/answers/*/signatures/similarnn/crop_features/*/rect/width": 1.0,
    "/answers/*/signatures/similarnn/crop_features/*/rect/height": 1.0,
    "/answers/*/signatures/similarnn/crop_features/*/image_features/*/features/*": 0.2,
    "": 2e-4
}

def convert_to_json(features):
    # This stupid function can be replaced with google.protobuf.json_format.MessageToJson
    # as soon as sandbox will be able to move from ancient protobuf 2.3.0
    result = {}
    image_features = []
    for item in features.ImageFeatures:
        image_features.append({"layer_name": item.LayerName, "features": list(item.Features)})
    result["image_features"] = image_features

    crop_features = []
    for crop in features.CropFeatures:
        rect = {"x": crop.Position.X, "y": crop.Position.Y, "width": crop.Position.Width, "height": crop.Position.Height}
        image_feat = [{"layer_name": item.LayerName, "features": list(item.Features)} for item in crop.ImageFeatures]
        crop_features.append({"rect": rect, "image_features": image_feat})
    result["crop_features"] = crop_features

    face_features = []
    for item in features.FaceFeatures:
        face_attrs = {"layer_name": item.LayerName}
        face_attrs["center"] = [item.CenterX, item.CenterY]
        face_attrs["size"] = [item.Width, item.Height]
        face_attrs["angle"] = item.Angle
        face_attrs["confidence"] = item.Confidence
        face_attrs["features"] = list(item.Features)
        face_features.append(face_attrs)
    result["face_features"] = face_features

    predictions = []
    for item in features.Predictions:
        predictions.append({"label": item.LabelName, "probability": item.Probability, "version": item.Version})
    result["predictions"] = predictions

    return result


def decode_similarnn_to_json(response):
    from extsearch.images.kernel.cbir.cbirdaemon.protos import similar_features_pb2
    version, data = response.split('@')
    raw = base64.b64decode(data)
    features = similar_features_pb2.TSimilarFeaturesPB()
    features.ParseFromString(raw)
    result = convert_to_json(features)
    result["version"] = version
    return result


def decode_cbirdaemon_subtasks_to_json(binary):
    from extsearch.images.daemons.cbirdaemon2.request_info.proto import rtconfig_pb2
    features = rtconfig.TEngineRequest()
    features.ParseFromString(binary)
    result = convert_to_json(features)
    return result


def decode_response(response):
    result = response
    if "answers" not in result:
        return {}
    for a in result["answers"]:
        if "signatures" in a:
            signatures = a["signatures"]
            if "similarnn" in signatures:
                signatures["similarnn"] = decode_similarnn_to_json(signatures["similarnn"])
        for s in a.get("signatures_vec", []):
            if s["name"] == "similarnn":
                s["value"] = decode_similarnn_to_json(s["value"])
        if a.get("type", "") == "cbirdaemon_subtasks":
            a["binary"] = decode_cbirdaemon_subtasks_to_json(a["binary"])
    return result


def _wrap_block(key, output):
    new_output = []
    if output:
        new_output.append(printers.HtmlBlock.start(key))
        new_output.extend(output)
        new_output.append(printers.HtmlBlock.end())
    return new_output


def compare_json(object1, object2, path = ""):

    def _print_block(key, value, color):
        if isinstance(value, dict):
            # subobject
            output = []
            output.append(printers.HtmlBlock.start(key, color))
            for k, v in value.iteritems():
                output.extend(_print_block(k, v, color))
            output.append(printers.HtmlBlock.end())
            return output
        elif isinstance(value, list):
            # sublist
            output = []
            output.append(printers.HtmlBlock.start(key, color))
            for i, v in enumerate(value):
                output.extend(_print_block("#{}".format(i), v, color))
            output.append(printers.HtmlBlock.end())
            return output
        elif isinstance(value, basestring):
            # simple key
            return [
                printers.HtmlBlock.start(key, color),
                printers.HtmlBlock.simple_data(value, text_bg_class=color),
                printers.HtmlBlock.end(),
            ]
        else:
            return []

    if object1 is None or object2 is None:
        if object1 is None and object2 is None:
            return []
        line1, line2 = printers.match_string_color_changed(str(object1), str(object2))
        return [
            printers.HtmlBlock.colored_data(line1),
            printers.HtmlBlock.colored_data(line2),
        ]
    elif isinstance(object1, dict) and isinstance(object2, dict):
        # compare keys
        keys1, keys2 = set(object1.keys()), set(object2.keys())
        output = []
        for key in keys2.difference(keys1):
            output.extend(_print_block(key, object2[key], printers.DiffType.ADDED))
        for key in keys1.difference(keys2):
            output.extend(_print_block(key, object1[key], printers.DiffType.REMOVED))
        for key in keys1.intersection(keys2):
            output.extend(_wrap_block(key, compare_json(object1[key], object2[key], "%s/%s" % (path, key))))
        return output
    elif isinstance(object1, list) and isinstance(object2, list):
        i = 0
        output = []
        for o1, o2 in itertools.izip_longest(sorted(object1), sorted(object2)):
            if o1 is not None and o2 is not None:
                output.extend(_wrap_block("#{}".format(i), compare_json(o1, o2, "%s/*" % (path))))
            elif o1 is None and o2 is not None:
                output.extend(_print_block("#{}".format(i), o2, printers.DiffType.ADDED))
            elif o1 is not None and o2 is None:
                output.extend(_print_block("#{}".format(i), o1, printers.DiffType.REMOVED))
            i += 1
        return output
    elif isinstance(object1, basestring) and isinstance(object2, basestring):
        # compare values
        if object1 != object2:
            line1, line2 = printers.match_string_color_changed(object1, object2)
            return [
                printers.HtmlBlock.colored_data(line1),
                printers.HtmlBlock.colored_data(line2),
            ]
        return []
    elif isinstance(object1, float) and isinstance(object2, float):
        eps = EPS_BY_PATH[path] if path in EPS_BY_PATH else EPS_BY_PATH[""]
        if abs(object1 - object2) > eps:
            line1, line2 = printers.match_string_color_changed(str(object1), str(object2))
            return [
                printers.HtmlBlock.colored_data(line1),
                printers.HtmlBlock.colored_data(line2),
            ]
        return []
    elif isinstance(object1, int) and isinstance(object2, int):
        # compare values
        eps = EPS_BY_PATH[path] if path in EPS_BY_PATH else EPS_BY_PATH[""]
        if abs(object1 - object2) > eps:
            line1, line2 = printers.match_string_color_changed(str(object1), str(object2))
            return [
                printers.HtmlBlock.colored_data(line1),
                printers.HtmlBlock.colored_data(line2),
            ]
        return []
    else:
        raise errors.SandboxTaskFailureError("Wrong objects to compare: {}, {}".format(object1, object2))


def compare(response1, response2, title, diff_path):
    decoded_response1 = decode_response(response1)
    decoded_response2 = decode_response(response2)
    with open(os.path.join(diff_path, "responses1.json"), "a") as resp1_file:
        resp1_file.write(json.dumps(decoded_response1) + "\n\n")
    with open(os.path.join(diff_path, "responses2.json"), "a") as resp2_file:
        resp2_file.write(json.dumps(decoded_response2) + "\n\n")
    return _wrap_block(title, compare_json(decoded_response1, decoded_response2))
