import itertools
import re
import os.path

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import task

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common.differ import printers
from sandbox.projects.images.daemons import resources as images_daemons_resources
from sandbox.projects.images.daemons.ImagesCompareCbirdaemon2ApphostResponses import compare_responses

_FEATURE_RE = re.compile(r"([^{]+){([^}]+)}")
_DIFF_COUNT_KEY = "diff_count"
_COMPARE_RESULT_KEY = "compare_result"


class Responses1Parameter(parameters.ResourceSelector):
    name = "cbir_daemon_responses1_resource_id"
    description = "Cbirdaemon responses #1"
    resource_type = images_daemons_resources.CBIR_DAEMON2_RESPONSES


class Responses2Parameter(parameters.ResourceSelector):
    name = "cbir_daemon_responses2_resource_id"
    description = "Cbirdaemon responses #2"
    resource_type = images_daemons_resources.CBIR_DAEMON2_RESPONSES


class ImagesCompareCbirdaemon2Responses(task.SandboxTask):
    """
        Get cbirdaemon responses
    """

    type = 'IMAGES_COMPARE_CBIRDAEMON2_RESPONSES'

    input_parameters = (Responses1Parameter, Responses2Parameter)

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        self.create_resource(
            self.descr,
            self.__get_diff_dir(),
            images_daemons_resources.CBIR_DAEMON2_RESPONSES_DIFF
        )

    def on_execute(self):
        diff_path = self.__get_diff_dir()
        paths.make_folder(diff_path, delete_content=True)

        def _print_block(key, content, color):
            return [
                printers.HtmlBlock.start(key, color),
                printers.HtmlBlock.simple_data(content, text_bg_class=color),
                printers.HtmlBlock.end(),
            ]

        def _print_change(key, content1, content2):
            color = printers.DiffType.CHANGED
            line1, line2 = printers.match_string_color_changed(content1, content2)
            return [
                printers.HtmlBlock.start(key, bg_class=color),
                printers.HtmlBlock.colored_data(line1),
                printers.HtmlBlock.colored_data(line2),
                printers.HtmlBlock.end(),
            ]

        diff_content = []
        iterator1 = self.__responses_iterator(self.ctx[Responses1Parameter.name])
        iterator2 = self.__responses_iterator(self.ctx[Responses2Parameter.name])
        for n, responses in enumerate(itertools.izip_longest(iterator1, iterator2)):
            response1, response2 = responses
            if response1 is None or response2 is None:
                raise errors.SandboxTaskFailureError("Different number of responses")

            output = compare_responses.compare(response1, response2, "response #{}".format(n), diff_path)
            if output:
                self.ctx[_DIFF_COUNT_KEY] = self.ctx.get(_DIFF_COUNT_KEY, 0) + 1
                diff_content.extend(output)

        if diff_content:
            printers.HtmlFile.write_companion(diff_path, "response_diff_style.css")
            printers.HtmlFile.write_companion(diff_path, "response_diff_scripts.js")
            with open(os.path.join(diff_path, "diff.html"), "w") as diff_file:
                printers.HtmlFile.write(diff_file, "".join(diff_content), buttons=False)
            self.ctx["compare_result"] = False
        else:
            fu.write_file(os.path.join(diff_path, utils.NO_DIFF_NAME), utils.no_diff_detected())
            self.ctx["compare_result"] = True

    def get_short_task_result(self):
        if _COMPARE_RESULT_KEY in self.ctx:
            return "no diff" if self.ctx[_COMPARE_RESULT_KEY] else "{} diff".format(self.ctx[_DIFF_COUNT_KEY])

    def __responses_iterator(self, responses_resource_id):
        responses_path = self.sync_resource(responses_resource_id)
        with open(responses_path) as responses_file:
            while True:
                header = responses_file.readline()
                if not header:
                    break
                size = int(header)
                data = responses_file.read(size)
                if len(data) != size:
                    raise errors.SandboxTaskFailureError("Invalid content in file '{}'".format(responses_path))
                yield self.__parse_response(data)

    def __parse_response(self, data):
        result = {}
        for line in data.strip().split("\n"):
            m = _FEATURE_RE.search(line)
            if not m:
                raise errors.SandboxTaskFailureError("Unsupported response format: '{}'".format(data))
            result[m.group(1)] = m.group(2)
        # make app-host compatible structure
        return {"answers": [{"signatures": result}]}

    def __get_diff_dir(self):
        return self.abs_path("diff")


__Task__ = ImagesCompareCbirdaemon2Responses
