import itertools
import os.path

from sandbox import sdk2
import sandbox.common.types.resource as ctr
import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import sandboxapi

import sandbox.projects.common.app_host.converter as apphost_converter
from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common.search import requester as search_requester
from sandbox.projects.common.differ import printers
from sandbox.projects.images.daemons import resources as images_daemons_resources

import compare_responses


class ImagesCompareRimpatchdaemonResponses(sdk2.Task):
    """
        Compare rimpatchdaemon responses
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC

    class Parameters(sdk2.Task.Parameters):
        rimpatchdaemon_responses1_resource_id = sdk2.parameters.Resource(
            "Rimpatchdaemon responses #1",
            resource_type=images_daemons_resources.IMAGES_RIMPATCHDAEMON_APPHOST_RESPONSES,
            state=(ctr.State.READY),
            required=True
        )
        rimpatchdaemon_responses2_resource_id = sdk2.parameters.Resource(
            "Rimpatchdaemon responses #2",
            resource_type=images_daemons_resources.IMAGES_RIMPATCHDAEMON_APPHOST_RESPONSES,
            state=(ctr.State.READY),
            required=True
        )

    def on_execute(self):
        diff_res = images_daemons_resources.IMAGES_RIMPATCHDAEMON_APPHOST_RESPONSES_DIFF(
            self,
            self.Parameters.description,
            'diff'
        )
        diff_path = str(sdk2.ResourceData(diff_res).path)
        paths.make_folder(diff_path, delete_content=True)

        iterator1 = self.__responses_iterator(self.Parameters.rimpatchdaemon_responses1_resource_id)
        iterator2 = self.__responses_iterator(self.Parameters.rimpatchdaemon_responses2_resource_id)

        diff_content = []
        for n, responses in enumerate(itertools.izip_longest(iterator1, iterator2)):
            response1, response2 = responses
            if response1 is None or response2 is None:
                raise errors.SandboxTaskFailureError("Different number of responses")

            output = compare_responses.compare(response1, response2, "response #{}".format(n))
            if output:
                self.Context.diff_count = 1 if not self.Context.diff_count else self.Context.diff_count + 1
                diff_content.extend(output)

        if diff_content:
            printers.HtmlFile.write_companion(diff_path, "response_diff_style.css")
            printers.HtmlFile.write_companion(diff_path, "response_diff_scripts.js")
            with open(os.path.join(diff_path, "diff.html"), "w") as diff_file:
                printers.HtmlFile.write(diff_file, "".join(diff_content), buttons=False)
            self.Context.compare_result = False
        else:
            fu.write_file(os.path.join(diff_path, utils.NO_DIFF_NAME), utils.no_diff_detected())
            self.Context.compare_result = True

    def get_short_task_result(self):
        if self.Context.compare_result:
            return "no diff" if self.Context.compare_result else "{} diff".format(self.Context.diff_count)

    def __responses_iterator(self, responses_resource):
        responses_path = str(sdk2.ResourceData(responses_resource).path)
        # TODO: don't use converter released with noapache
        converter_path = str(sdk2.ResourceData(
            resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE.
                find(
                    arc=sandboxapi.ARCH_LINUX,
                    state=(ctr.State.READY),
                    released='stable'
                ).order(-sdk2.Resource.id).first()
        ).path)
        converter_func = apphost_converter.convert_input_response
        with open(responses_path) as responses_file:
            for data in search_requester.sequence_binary_data_from_stream(responses_file):
                yield converter_func(converter_path, data)
