import base64
import logging
import json
import string
import random

from numpy.random import choice

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka
from sandbox.projects.common import string as su
from sandbox.projects.common import utils
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.common.search import requester_compat as search_requester
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.thumbdaemon import utils as thumbdaemon_utils
from sandbox.projects.images.daemons import components as daemons_components
from sandbox.projects.images.resources import task as resources_task
from sandbox.projects.images.resources import fuzzing as resources_fuzzing


NAILDAEMON_PARAMS = daemons_components.create_naildaemon_params()
_SIGNATURE_TYPES = ["configurable_v2"]
_POST_REQUEST_TYPE = "post-synthetic"
_APPHOST_REQUEST_TYPE = "apphost-synthetic"
_REQUEST_TYPES = (_POST_REQUEST_TYPE, _APPHOST_REQUEST_TYPE)
_OUT_RESOURCE_ID = "{}_{}_out_resource_id"

_PLAN_RESOURCE_TYPE = "plan"
_QUERIES_RESOURCE_TYPE = "queries"

_CORPUS_SIZE = 10  # cbirdaemon too slow to eat large corpus

_CBIRDAEMON_REQUESTS = {
    "items" : {"main" : 0.75, "detector" : 0.05, "detector_alice" : 0.15, "detected_obj" : 0.05},
    "main" : {
        "ImageFeatures" : [
            {
                "Name" : "FeatV10"
            }
        ],
        "ImageCropFeatures" : [
            {
                "Name" : "FeatCropV10"
            }
        ],
        "FaceFeatures" : [
            {
                "Name" : "simfaces_ver5"
            }
        ],
        "Image2TextFeatures" : [
            {
                "Name" : "I2TVer11_v10"
            }
        ],
        "CbirFeatures" : {
            "Dscr" : False,
            "Quant" : True
        },
        "ImageInfo" : {
        },
        "ImageClassification" : {
            "Name" : "Classification10"
        },
        "AutoClassification" : {
            "Name" : "AutoVer10"
        },
        "Barcode":{
        }
    },
    "detector" : {
        "ObjectDetections" : {
            "Detector": [
                {
                    "Name" : "clothes",
                },
                {
                    "Name" : "interior",
                }
            ]
        }
    },
    "detector_alice" : {
        "ObjectDetections" : {
            "Detector": [
                {
                    "Name" : "clothes_alice",
                }
            ]
        },
        "ImageInfo" : {
        }
    },
    "detected_obj" : {
        "ImageFeatures" : [
            {
                "Name" : "ClothesV8",
            }
        ],
        "Image2TextFeatures" : [
            {
                "Name" : "I2TVer11_v10"
            }
        ],
        "ImageInfo" : {
        }
    }
}

class RequestLimitParameter(parameters.SandboxIntegerParameter):
    name = 'request_limit'
    description = 'Requests limit:'
    default_value = 100000


class AttributesParameter(parameters.SandboxStringParameter):
    name = 'attributes'
    description = 'Set additional attrs to resources (ex.: attr1=v1, attr2=v2)'
    do_not_copy = True


class CbirdaemonParameter(parameters.SandboxStringParameter):
    name = 'cbirdaemon_parameters'
    description = 'Requests parameters for cbirdaemon'
    default_value = json.dumps(_CBIRDAEMON_REQUESTS, indent=1)


class ImagesGenerateCbirdaemon2Requests(resources_task.ImagesProductionResourcesTask,
                                        resources_fuzzing.GenerateFuzzingTask,
                                        task.SandboxTask):
    """
        Generate cbirdaemon queries

        Use thumbs as input queries for cbirdaemon
    """

    type = "IMAGES_GENERATE_CBIRDAEMON2_REQUESTS"

    input_parameters = \
        (RequestLimitParameter, AttributesParameter, CbirdaemonParameter) + \
        resources_fuzzing.GenerateFuzzingTask.input_parameters + \
        NAILDAEMON_PARAMS.params + \
        search_requester.create_params()

    def _fuzzing_dir(self):
        return "extsearch/images/daemons/cbirdaemon2/calcsigs/fuzzing"

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        resources_fuzzing.GenerateFuzzingTask.on_enqueue(self)

        attributes = su.parse_attrs(self.ctx[AttributesParameter.name])
        for request_type in _REQUEST_TYPES:
            request_attributes = {k.format(request_type): v for k, v in attributes.iteritems()}
            self.ctx[_OUT_RESOURCE_ID.format(request_type, _QUERIES_RESOURCE_TYPE)] = self.create_resource(
                "{}, {}".format(self.descr, request_type),
                self.__get_requests_path(request_type),
                dolbilka_resources.DOLBILKA_STPD_QUERIES,
                arch=sandboxapi.ARCH_ANY,
                attributes=request_attributes
            ).id
            self.ctx[_OUT_RESOURCE_ID.format(request_type, _PLAN_RESOURCE_TYPE)] = self.create_resource(
                "{}, {}".format(self.descr, request_type),
                self.__get_plan_path(request_type),
                resource_types.BASESEARCH_PLAN,
                arch=sandboxapi.ARCH_ANY,
                attributes=request_attributes
            ).id

    def on_execute(self):
        thumbdaemon = daemons_components.get_naildaemon()
        json_requests_path = self.abs_path("cbirdaemon-json-requests.txt")

        cbirdaemon_requests = _CBIRDAEMON_REQUESTS
        if CbirdaemonParameter.name in self.ctx and\
                        self.ctx[CbirdaemonParameter.name] is not None and\
                        len(self.ctx[CbirdaemonParameter.name]) > 0:
            cbirdaemon_requests = json.loads(self.ctx[CbirdaemonParameter.name])

        self._fuzzing_corpus_init()
        with thumbdaemon:
            queries_iterator = self.__generate_queries(thumbdaemon)
            with open(self.__get_requests_path(_POST_REQUEST_TYPE), "w") as post_requests_file:
                with open(json_requests_path, "w") as json_requests_file:
                    empty = True
                    for _, status, image_data in search_requester.save_responses(self.ctx, queries_iterator):
                        if not status:
                            raise errors.SandboxTaskFailureError("Error during request processing: {}".format(image_data))
                        item = choice(cbirdaemon_requests["items"].keys(), 1, p=cbirdaemon_requests["items"].values())[0]
                        req = cbirdaemon_requests[item]
                        # post requests
                        content = _post_signature_request(image_data, req)
                        post_requests_file.write("{}\n".format(len(content)))
                        post_requests_file.write(content)

                        # apphost requests
                        content = _apphost_signature_request(image_data, req)
                        if not empty:
                            json_requests_file.write("\n")
                        json_requests_file.write("{}".format(content))
                        empty = False

                        # fuzzing corpus
                        self._fuzzing_corpus_add(image_data, max_corpus_size=_CORPUS_SIZE)

        # covert apphost requests to binary format
        app_host_tools_bundle = self.sync_resource(utils.get_and_check_last_resource_with_attribute(
            resource_types.APP_HOST_TOOLS_BUNDLE,
            attr_name="released",
            attr_value="stable"
        ).id)
        process.run_process(
            [app_host_tools_bundle + "/make_tank_ammo", "-i", json_requests_path, "-o", self.__get_requests_path(_APPHOST_REQUEST_TYPE)],
            outputs_to_one_file=False,
            log_prefix="make_tank_ammo"
        )

        # generate plans for dolbilka
        for request_type in _REQUEST_TYPES:
            dolbilka.convert_queries_to_plan(
                self.__get_requests_path(request_type),
                self.__get_plan_path(request_type),
                loader_type='phantom'
            )

        # link resources with current database
        index_type = search_settings.INDEX_CBIR_DAEMON
        attributes = {
            search_settings.SHARD_INSTANCE_ATTRIBUTE_NAME: self._get_basesearch_shard_name(index_type),
        }
        for request_type in _REQUEST_TYPES:
            for resource_type in (_PLAN_RESOURCE_TYPE, _QUERIES_RESOURCE_TYPE):
                resource_id = self.ctx[_OUT_RESOURCE_ID.format(request_type, resource_type)]
                utils.set_resource_attributes(resource_id, attributes)

        # update fuzzing resources
        try:
            self._fuzzing_corpus_finish(self._get_basesearch_database(index_type))
        except Exception as e:
            logging.info("Failed to commit fuzzing updates: {}".format(e))

    def __generate_queries(self, component):
        database_resource_id = self.ctx[NAILDAEMON_PARAMS.Database.name]
        request_limit = self.ctx[RequestLimitParameter.name]
        for n, thumb_id in enumerate(thumbdaemon_utils.get_thumb_ids(database_resource_id), 1):
            yield "http://localhost:{}/i?id={}&n=33&h=320&w=320".format(component.port, thumb_id)
            if request_limit and n == request_limit:
                break

    def __get_requests_path(self, request_type):
        return self.abs_path("cbirdaemon-{}-requests.txt".format(request_type))

    def __get_plan_path(self, request_type):
        return self.abs_path("cbirdaemon-{}-requests.plan".format(request_type))


def _http_payload(headers, data):
    output = []
    output.extend("{}: {}".format(k, v) for k, v in headers)
    output.append("")
    if data:
        output.append(data)
    return output


def _formdata_payload(boundary, field_name, content_type, content_data):
    headers = [
        ('Content-Type', '{}'.format(content_type)),
        ('Content-Disposition', 'form-data; name="{}"'.format(field_name)),
    ]
    return ["--{}".format(boundary)] + _http_payload(headers, content_data)


def _post_signature_request(image_data, req):
    boundary = "".join(random.sample(string.digits + string.lowercase, 30))

    data = []
    data.extend(_formdata_payload(boundary, "upfile", "image/jpeg", image_data))
    data.extend(["--{}--".format(boundary), ""])
    data = "\r\n".join(data)

    headers = [
        ("Sigrequest", ",".join(_SIGNATURE_TYPES)),
        ("RequestInfo", json.dumps(req)),
        ("Content-Type", "multipart/form-data; boundary={}".format(boundary)),
        ("Content-Length", len(data))
    ]
    query = "POST / HTTP/1.1"
    return "\r\n".join([query] + _http_payload(headers, data))


def _apphost_signature_request(image_data, req):
    query = {
        "answers": [{
            "__content_type": "json",
            "type": "cbirdaemon_params",
            "source": "CBIRDAEMON",
            "binary": {
                "image": base64.b64encode(image_data),
                "signatures": _SIGNATURE_TYPES,
                "request_info": req,
                "type": "cbirdaemon_params"
            }
        }]
    }
    return json.dumps(query)


__Task__ = ImagesGenerateCbirdaemon2Requests
