import logging
import random
import re

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka
from sandbox.projects.common import string
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.thumbdaemon import utils as thumbdaemon_utils
from sandbox.projects.images.resources import task as resources_task
from sandbox.projects.images.resources import fuzzing as resources_fuzzing


_SIGNATURE_REQUEST_RATIO = 100
_SIGNATURE_FLAGS = 2**15 - 2  # Turn on all possible signatures, except build-info-related
_SHARD_INSTANCE_ATTR = search_settings.SHARD_INSTANCE_ATTRIBUTE_NAME


class DatabaseParameter(parameters.ResourceSelector):
    name = 'database_resource_id'
    description = 'Database:'
    resource_type = resource_types.IMAGES_SEARCH_DATABASE


class RequestLimitParameter(parameters.SandboxIntegerParameter):
    name = 'request_limit'
    description = 'Requests limit:'
    default_value = 100000


class AttributesParameter(parameters.SandboxStringParameter):
    name = 'attributes'
    description = 'Set additional attrs to resources (ex.: attr1=v1, attr2=v2)'
    do_not_copy = True


class ImagesGenerateNaildaemonRequests(resources_task.ImagesProductionResourcesTask,
                                       resources_fuzzing.GenerateFuzzingTask,
                                       task.SandboxTask):
    """
        Generates queries for Nail daemon
    """
    type = 'IMAGES_GENERATE_NAILDAEMON_REQUESTS'

    input_parameters = (
        DatabaseParameter,
        RequestLimitParameter,
        AttributesParameter
    ) + resources_fuzzing.GenerateFuzzingTask.input_parameters

    def _fuzzing_dir(self):
        return "extsearch/images/daemons/naildaemon/lib/fuzzing"

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        resources_fuzzing.GenerateFuzzingTask.on_enqueue(self)

        database_resource = channel.channel.sandbox.get_resource(self.ctx[DatabaseParameter.name])
        attributes = string.parse_attrs(self.ctx[AttributesParameter.name])
        attributes.update({_SHARD_INSTANCE_ATTR: database_resource.attributes[_SHARD_INSTANCE_ATTR]})
        self.create_resource(
            self.descr,
            self.__get_requests_path(),
            resource_types.PLAIN_TEXT_QUERIES,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        )
        self.create_resource(
            self.descr,
            self.__get_plan_path(),
            resource_types.BASESEARCH_PLAN,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        )

    def on_execute(self):
        database_resource = channel.channel.sandbox.get_resource(self.ctx[DatabaseParameter.name])
        random.seed(database_resource.file_md5)

        thumb_config_resource_id = self._get_basesearch_config(search_settings.INDEX_THUMB_QUICK)
        thumb_config_local_path = self.sync_resource(thumb_config_resource_id)

        # Get all transformations supported by thumbdaemon
        # Getting from trunk is good enough - non-existent transforms will be handled as default
        transforms = []
        with open(thumb_config_local_path, 'r') as thumb_config_file:
            for line in thumb_config_file:
                if "ThumbChunk" in line:
                    transform = int(re.match(r'ThumbChunk:(\d+)=\d+', line).group(1))
                    transforms.append(transform)
        if not transforms:
            raise errors.SandboxTaskFailureError("Could not extract transforms from config")

        self._fuzzing_corpus_init()
        with open(self.__get_requests_path(), 'w') as result_file:
            max_n = self.ctx[RequestLimitParameter.name]
            for n, thumb_id in enumerate(thumbdaemon_utils.get_thumb_ids(self.ctx[DatabaseParameter.name])):
                if n > max_n:
                    break

                if random.randrange(_SIGNATURE_REQUEST_RATIO) % _SIGNATURE_REQUEST_RATIO == 0:
                    query = "/i?id={}&sigs={}".format(thumb_id, _SIGNATURE_FLAGS)
                else:
                    thumb_transform = random.choice(transforms)
                    query = "/i?id={}&n={}".format(thumb_id, thumb_transform)

                result_file.write(query + "\n")
                self._fuzzing_corpus_add(query)

        dolbilka.convert_queries_to_plan(self.__get_requests_path(), self.__get_plan_path())
        try:
            self._fuzzing_corpus_finish(self.ctx[DatabaseParameter.name])
        except Exception as e:
            logging.info("Failed to commit fuzzing updates: {}".format(e))

    def __get_requests_path(self):
        return self.abs_path("thumb-requests.txt")

    def __get_plan_path(self):
        return self.abs_path("thumb-requests.plan")


__Task__ = ImagesGenerateNaildaemonRequests
